% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/amova.r
\name{poppr.amova}
\alias{poppr.amova}
\alias{amova}
\title{Perform Analysis of Molecular Variance (AMOVA) on genind or genclone objects.}
\usage{
poppr.amova(x, hier = NULL, clonecorrect = FALSE, within = TRUE,
  dist = NULL, squared = TRUE, freq = TRUE, correction = "quasieuclid",
  sep = "_", filter = FALSE, threshold = 0,
  algorithm = "farthest_neighbor", threads = 1L, missing = "loci",
  cutoff = 0.05, quiet = FALSE, method = c("ade4", "pegas"), nperm = 0)
}
\arguments{
\item{x}{a \link[=genind-class]{genind}, \link[=genclone-class]{genclone}, \link[=genlight-class]{genlight}, or \link[=snpclone-class]{snpclone} object}

\item{hier}{a hierarchical \link[=formula]{formula} that defines your population
hierarchy. (e.g.: \code{~Population/Subpopulation}). \strong{See Details below}.}

\item{clonecorrect}{\code{logical} if \code{TRUE}, the data set will be clone corrected
with respect to the lowest level of the hierarchy. The default is set to
\code{FALSE}. See \code{\link[=clonecorrect]{clonecorrect()}} for details.}

\item{within}{\code{logical}. When this is set to \code{TRUE} (Default), variance
within individuals are calculated as well. If this is set to \code{FALSE}, The
lowest level of the hierarchy will be the sample level. See Details below.}

\item{dist}{an optional distance matrix calculated on your data. If this is
set to \code{NULL} (default), the raw pairwise distances will be calculated via
\code{\link[=dist]{dist()}}.}

\item{squared}{if a distance matrix is supplied, this indicates whether or
not it represents squared distances.}

\item{freq}{\code{logical}. If \code{within = FALSE}, the parameter rho is calculated
(Ronfort et al. 1998; Meirmans and Liu 2018). By setting \code{freq = TRUE},
(default) allele counts will be converted to frequencies before the
distance is calculated, otherwise, the distance will be calculated on
allele counts, which can bias results in mixed-ploidy data sets. Note that
this option has no effect for haploid or presence/absence data sets.}

\item{correction}{a \code{character} defining the correction method for
non-euclidean distances. Options are \code{\link[ade4:quasieuclid]{ade4::quasieuclid()}} (Default),
\code{\link[ade4:lingoes]{ade4::lingoes()}}, and \code{\link[ade4:cailliez]{ade4::cailliez()}}. See Details below.}

\item{sep}{Deprecated. As of poppr version 2, this argument serves no
purpose.}

\item{filter}{\code{logical} When set to \code{TRUE}, mlg.filter will be run to
determine genotypes from the distance matrix. It defaults to \code{FALSE}. You
can set the parameters with \code{algorithm} and \code{threshold} arguments. Note
that this will not be performed when \code{within = TRUE}. Note that the
threshold should be the number of allowable substitutions if you don't
supply a distance matrix.}

\item{threshold}{a number indicating the minimum distance two MLGs must be
separated by to be considered different. Defaults to 0, which will reflect
the original (naive) MLG definition.}

\item{algorithm}{determines the type of clustering to be done. 
\describe{
  \item{"farthest_neighbor"}{\emph{ (default) }merges clusters based on the 
  maximum distance between points in either cluster. This is the strictest of
  the three.}
  \item{"nearest_neighbor"}{ merges clusters based on the minimum distance
  between points in either cluster. This is the loosest of the three.}
  \item{"average_neighbor"}{ merges clusters based on the average distance
  between every pair of points between clusters.}
}}

\item{threads}{\code{integer} When using filtering or genlight objects, this
parameter specifies the number of parallel processes passed to
\code{\link[=mlg.filter]{mlg.filter()}} and/or \code{\link[=bitwise.dist]{bitwise.dist()}}.}

\item{missing}{specify method of correcting for missing data utilizing
options given in the function \code{\link[=missingno]{missingno()}}. Default is \code{"loci"}. This only
applies to genind or genclone objects.}

\item{cutoff}{specify the level at which missing data should be
removed/modified. See \code{\link[=missingno]{missingno()}} for details. This only applies to
genind or genclone objects.}

\item{quiet}{\code{logical} If \code{FALSE} (Default), messages regarding any
corrections will be printed to the screen. If \code{TRUE}, no messages will be
printed.}

\item{method}{Which method for calculating AMOVA should be used? Choices
refer to package implementations: "ade4" (default) or "pegas". See details
for differences.}

\item{nperm}{the number of permutations passed to the pegas implementation of
amova.}
}
\value{
a list of class \code{amova} from the ade4 or pegas package. See
\code{\link[ade4:amova]{ade4::amova()}} or \code{\link[pegas:amova]{pegas::amova()}} for details.
}
\description{
This function simplifies the process necessary for performing AMOVA in R. It
gives user the choice of utilizing either the \pkg{ade4} or the \pkg{pegas}
implementation of AMOVA. See \code{\link[ade4:amova]{ade4::amova()}} (ade4) and \code{\link[pegas:amova]{pegas::amova()}}
(pegas) for details on the specific implementation.
}
\details{
The poppr implementation of AMOVA is a very detailed wrapper for the
ade4 implementation. The output is an \code{\link[ade4:amova]{ade4::amova()}} class list that
contains the results in the first four elements. The inputs are contained
in the last three elements. The inputs required for the ade4 implementation
are:
\enumerate{
\item a distance matrix on all unique genotypes (haplotypes)
\item a data frame defining the hierarchy of the distance matrix
\item a genotype (haplotype) frequency table.
}

All of this data can be constructed from a \link[=genind-class]{genind} or
\link[=genlight-class]{genlight} object, but can be daunting for a novice R user.
\emph{This function automates the entire process}. Since there are many
variables regarding genetic data, some points need to be highlighted:

\subsection{On Hierarchies:}{The hierarchy is defined by different
population strata that separate your data hierarchically. These strata are
defined in the \strong{strata} slot of \link[=genind-class]{genind},
\link[=genlight-class]{genlight}, \link[=genclone-class]{genclone}, and
\link[=snpclone-class]{snpclone} objects. They are useful for defining the
population factor for your data. See the function \code{\link[=strata]{strata()}} for details on
how to properly define these strata.}

\subsection{On Within Individual Variance:}{ Heterozygosities within
genotypes are sources of variation from within individuals and can be
quantified in AMOVA. When \code{within = TRUE}, poppr will split genotypes into
haplotypes with the function \code{\link[=make_haplotypes]{make_haplotypes()}} and use those to calculate
within-individual variance. No estimation of phase is made. This acts much
like the default settings for AMOVA in the Arlequin software package.
Within individual variance will not be calculated for haploid individuals
or dominant markers as the haplotypes cannot be split further. Setting
\code{within = FALSE} uses the euclidean distance of the allele frequencies
within each individual. \strong{Note:} \code{within = TRUE} is incompatible with
\code{filter = TRUE}. In this case, \code{within} will be set to \code{FALSE}}

\subsection{On Euclidean Distances:}{ With the \pkg{ade4} implementation of
AMOVA (utilized by \pkg{poppr}), distances must be Euclidean (due to the
nature of the calculations). Unfortunately, many genetic distance measures
are not always euclidean and must be corrected for before being analyzed.
Poppr automates this with three methods implemented in \pkg{ade4},
\code{\link[=quasieuclid]{quasieuclid()}}, \code{\link[=lingoes]{lingoes()}}, and \code{\link[=cailliez]{cailliez()}}. The correction of these
distances should not adversely affect the outcome of the analysis.}

\subsection{On Filtering:}{ Filtering multilocus genotypes is performed by
\code{\link[=mlg.filter]{mlg.filter()}}. This can necessarily only be done AMOVA tests that do not
account for within-individual variance. The distance matrix used to
calculate the amova is derived from using \code{\link[=mlg.filter]{mlg.filter()}} with the option
\code{stats = "distance"}, which reports the distance between multilocus
genotype clusters. One useful way to utilize this feature is to correct for
genotypes that have equivalent distance due to missing data. (See example
below.)}

\subsection{On Methods:}{ Both \pkg{ade4} and \pkg{pegas} have
implementations of AMOVA, both of which are appropriately called "amova".
The ade4 version is faster, but there have been questions raised as to the
validity of the code utilized. The pegas version is slower, but careful
measures have been implemented as to the accuracy of the method. It must be
noted that there appears to be a bug regarding permuting analyses where
within individual variance is accounted for (\code{within = TRUE}) in the pegas
implementation. If you want to perform permutation analyses on the pegas
implementation, you must set \code{within = FALSE}. In addition, while clone
correction is implemented for both methods, filtering is only implemented
for the ade4 version.}

\subsection{On Polyploids:}{ As of \pkg{poppr} version 2.7.0, this
function is able to calculate phi statistics for within-individual variance
for polyploid data with \strong{full dosage information}. When a data set does
not contain full dosage information for all samples, then the resulting
pseudo-haplotypes will contain missing data, which would result in an
incorrect estimate of variance.

Instead, the AMOVA will be performed on the distance matrix derived from
allele counts or allele frequencies, depending on the \code{freq} option. This
has been shown to be robust to estimates with mixed ploidy (Ronfort et al.
1998; Meirmans and Liu 2018). If you wish to brute-force your way to
estimating AMOVA using missing values, you can split your haplotypes with
the \code{\link[=make_haplotypes]{make_haplotypes()}} function.

One strategy for addressing ambiguous dosage in your polyploid data set
would be to convert your data to \pkg{polysat}'s \code{genambig} class with the
\code{\link[=as.genambig]{as.genambig()}}, estimate allele frequencies with \code{\link[polysat:deSilvaFreq]{polysat::deSilvaFreq()}},
and use these frequencies to randomly sample alleles to fill in the
ambiguous alleles.
}
}
\examples{
data(Aeut)
strata(Aeut) <- other(Aeut)$population_hierarchy[-1]
agc <- as.genclone(Aeut)
agc
amova.result <- poppr.amova(agc, ~Pop/Subpop)
amova.result
amova.test <- randtest(amova.result) # Test for significance
plot(amova.test)
amova.test

\dontrun{

# You can get the same results with the pegas implementation
amova.pegas <- poppr.amova(agc, ~Pop/Subpop, method = "pegas")
amova.pegas
amova.pegas$varcomp/sum(amova.pegas$varcomp)

# Clone correction is possible
amova.cc.result <- poppr.amova(agc, ~Pop/Subpop, clonecorrect = TRUE)
amova.cc.result
amova.cc.test <- randtest(amova.cc.result)
plot(amova.cc.test)
amova.cc.test


# Example with filtering
data(monpop)
splitStrata(monpop) <- ~Tree/Year/Symptom
poppr.amova(monpop, ~Symptom/Year) # gets a warning of zero distances
poppr.amova(monpop, ~Symptom/Year, filter = TRUE, threshold = 0.1) # no warning


}
}
\references{
Excoffier, L., Smouse, P.E. and Quattro, J.M. (1992) Analysis of
molecular variance inferred from metric distances among DNA haplotypes:
application to human mitochondrial DNA restriction data. \emph{Genetics},
\strong{131}, 479-491.

Ronfort, J., Jenczewski, E., Bataillon, T., and Rousset, F. (1998). Analysis
of population structure in autotetraploid species. \emph{Genetics}, \strong{150},
921–930.

Meirmans, P., Liu, S. (2018) Analysis of Molecular Variance (AMOVA) for
Autopolyploids \emph{Submitted}.
}
\seealso{
\code{\link[ade4:amova]{ade4::amova()}}, \code{\link[pegas:amova]{pegas::amova()}}, \code{\link[=clonecorrect]{clonecorrect()}}, \code{\link[=diss.dist]{diss.dist()}},
\code{\link[=missingno]{missingno()}}, \code{\link[ade4:is.euclid]{ade4::is.euclid()}}, \code{\link[=strata]{strata()}}, \code{\link[=make_haplotypes]{make_haplotypes()}},
\code{\link[=as.genambig]{as.genambig()}}
}
\keyword{amova}
