% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/round_robin.R
\name{rraf}
\alias{rraf}
\title{Round Robin Allele Frequencies}
\usage{
rraf(gid, pop = NULL, res = "list", by_pop = FALSE, correction = TRUE)
}
\arguments{
\item{gid}{a genind or genclone object}

\item{pop}{either a formula to set the population factor from the
\code{\link{strata}} slot or a vector specifying the population factor for
each sample. Defaults to \code{NULL}.}

\item{res}{Either "list" (default), "vector", or "data.frame".}

\item{by_pop}{When this is \code{TRUE}, the calculation will be done by
population. Defaults to \code{FALSE}}

\item{correction}{a logical indicating whether or not zero value allele 
frequencies should be set to 1/n (Default: \code{TRUE})}
}
\value{
a vector or list of allele frequencies
}
\description{
This function utilizes \code{\link{rrmlg}} to calculate multilocus genotypes 
and then subsets each locus by the resulting MLGs to calculate the 
round-robin allele frequencies used for pgen and psex.
}
\details{
Calculating allele frequencies for clonal populations is a difficult
  task. Frequencies calculated on non-clone-corrected data suffer from bias
  due to non-independent samples. On the other hand, frequencies calculated
  on clone-corrected data artificially increases the significance of rare
  alleles. The method of round-robin allele frequencies as presented in Parks
  and Werth (1993) provides a method of calculating allele frequencies in a
  way that minimizes both of these effects. Allele frequencies at a given
  locus are calculated based on samples that are \strong{clone corrected
  without that locus}. This means that rare alleles might not be represented,
  giving them an allele frequency of "0". For some analyses, this value is
  perfectly fine, but for analyses such as \code{\link{pgen}} and
  \code{\link{psex}}, this could result in genotype probabilities of "0". One
  remedy for this is in the \code{correction} option. This will replace any 
  zero-frequency alleles by 1/n, but it will mean that the sum of allele
  frequencies at that locus will no longer be 1.
}
\note{
When \code{by_pop = TRUE}, the output will be a matrix of allele 
  frequencies. Additionally, when the argument \code{pop} is not \code{NULL},
  \code{by_pop} is automatically \code{TRUE}. When \code{correction = TRUE},
  \emph{n} refers to the population size.
}
\examples{

data(Pram)
# Round robin allele frequencies.
rraf(Pram)


\dontrun{

## Round robin allele frequencies will be different than observed

# Compare to without round robin:
PrLoc <- seploc(Pram, res = "mat") # get locus by matrix
lapply(PrLoc, colMeans, na.rm = TRUE)

# Without round robin, clone corrected:
Pcc <- clonecorrect(Pram, strata = NA) # indiscriminantly clone correct
PccLoc <- seploc(Pcc, res = "mat")
lapply(PccLoc, colMeans, na.rm = TRUE)

## Different methods of obtaining round robin allele frequencies

# Get vector output.
rraf(Pram, res = "vector")

# Get data frame output and plot.
(Prdf <- rraf(Pram, res = "data.frame"))
library("ggplot2")
ggplot(Prdf, aes(y = allele, x = frequency)) +
  geom_point() +
  facet_grid(locus ~ ., scale = "free_y", space = "free")

## Round Robin allele frequencies by populations (matrix only)

# Get frequencies per population without correction for zero-frequency alleles
(Prbp <- rraf(Pram, by_pop = TRUE, correction = FALSE))
# You can set the zero-frequency alleles to be 1/nInd(Pram)
Prbp[Prbp == 0] <- 1/nInd(Pram)
Prbp

# Get frequencies per population, but set zero-frequency alleles to 1/n
(Prbp <- rraf(Pram, by_pop = TRUE, correction = TRUE))


}
}
\author{
Zhian N. Kamvar, Jonah Brooks, Stacy A. Krueger-Hadfield, Erik Sotka
}
\references{
Arnaud-Haond, S., Duarte, C. M., Alberto, F., & Serrão, E. A. 2007.
Standardizing methods to address clonality in population studies.
\emph{Molecular Ecology}, 16(24), 5115-5139.

Parks, J. C., & Werth, C. R. 1993. A study of spatial features of clones in a
population of bracken fern, \emph{Pteridium aquilinum} (Dennstaedtiaceae).
\emph{American Journal of Botany}, 537-544.
}
\seealso{
\code{\link{rrmlg}}, \code{\link{pgen}}, \code{\link{psex}}
}

