#' Visualize Ratio of Variance of Each Pooled Measurement to Variance of Each
#' Unpooled Measurement as Function of Pool Size
#'
#' Useful for determining whether pooling is a good idea, and finding the
#' optimal pool size if it is.
#'
#'
#' @inheritParams poolcost_t
#'
#'
#' @return Plot generated by \code{\link[ggplot2]{ggplot}}.
#'
#'
#' @examples
#' # Plot ratio of variances vs. pool size with default settings
#' poolvar_t()
#'
#' # Add processing error and other per-subject costs
#' poolvar_t(sigsq_p = 0.2, other_costs = 0.1)
#'
#'
#'@export
poolvar_t <- function(g = 1: 10,
                      sigsq = 1,
                      sigsq_p = 0,
                      sigsq_m = 0,
                      multiplicative = FALSE,
                      mu = 1,
                      type = "two.sample",
                      assay_cost = 100,
                      other_costs = 0,
                      labels = TRUE,
                      ylim = NULL) {

  # Prepare data for ggplot
  if (multiplicative) {
    sigsq_pm <- sigsq_p + sigsq_m + sigsq_p * sigsq_m
    var_ratio <- (sigsq + sigsq_m + sigsq * sigsq_m) /
      (sigsq_pm * (mu^2 + sigsq / g) + sigsq / g)
  } else {
    var_ratio <- (sigsq + sigsq_m) / (sigsq / g + sigsq_p + sigsq_m)
  }
  var_ratio_adj <- var_ratio /
    ((assay_cost + g * other_costs) / (assay_cost + other_costs))
  df <- data.frame(g = g,
                   var_ratio = var_ratio,
                   var_ratio_adj = var_ratio_adj)
  df$max <- 0
  df$max[which.max(df$var_ratio_adj)] <- 1

  # Create plot
  p <- ggplot(df, aes(g, var_ratio_adj)) +
    geom_point() +
    labs(title = "Ratio of Variances, Traditional vs. Pooled",
         y = "Ratio, adjusted for total costs/assay",
         x = "Pool size") +
    geom_hline(yintercept = 1, linetype = 2) +
    theme_bw() +
    theme(panel.grid.major = element_blank(),
          panel.grid.minor = element_blank())
    #theme(legend.title = element_blank())

  # Label max
  if (labels) {
    p <- p + geom_label_repel(
      data = subset(df, max == 1),
      aes(g, var_ratio_adj,
          label = paste(round(var_ratio_adj, 1), "(g = ", g, ")", sep = "")),
      box.padding = 0.5,
      point.padding = 0.3,
      label.padding = 0.4
    )
  }

  # Adjust ylim
  if (! is.null(ylim)) {
    p <- p + ylim(ylim)
  }
  p

}
poolvar_t()
poolvar_t(sigsq_p = 0.2, other_costs = 10)
