% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/benchmark.R
\name{pdSplineReg}
\alias{pdSplineReg}
\title{Cubic smoothing spline regression for HPD matrices}
\usage{
pdSplineReg(P, f0, lam = 1, Nd, ini.step = 1, max.iter = 100,
  eps = 0.001, ...)
}
\arguments{
\item{P}{a \eqn{(d,d,n)}-dimensional array corresponding to a sequence of observed noisy HPD matrices.}

\item{f0}{a \eqn{(d,d,n)}-dimensional array corresponding to an initial HPD curve estimator of the smooth
target curve of HPD matrices.}

\item{lam}{a smoothness penalty parameter, defaults to \code{lam = 1}. If \code{lam = 0}, the penalized regression estimator
coincides with geodesic interpolation of the data points. If \code{lam} increases to \eqn{\infty},
the penalized regression estimator is approximately a fitted geodesic curve.}

\item{Nd}{a numeric value (\code{Nd <= n}) determining a lower resolution of the cubic spline regression estimator to speed up
computations, defaults to the number of data observations \eqn{n}.}

\item{ini.step}{initial candidate step size in a backtracking line search based on the Armijo-Goldstein
condition, defaults to \code{ini.step = 1}.}

\item{max.iter}{maximum number of gradient descent iterations, defaults to \code{max.iter = 100}.}

\item{eps}{stopping criterion. The gradient descent procedure exits if the absolute difference of the
of the evaluated objective function is smaller than \code{eps}, defaults to \code{eps = 1E-3}.}

\item{...}{additional arguments for internal usage.}
}
\value{
A list with three components:
\item{f }{a \eqn{(d, d, N_d)}-dimensional array containing the HPD cubic smoothing spline regression estimator.}
\item{cost }{a numeric vector containing the costs of the objective function after each gradient descent iteration.}
\item{total.iter }{total number of gradient descent iterations.}
}
\description{
\code{pdSplineReg()} performs cubic smoothing spline regression on the Riemannian manifold of
HPD matrices equipped with the Riemannian metric through minimization of a penalized regression
objective function using a geometric conjugate gradient descent method as outlined in (Boumal and Absil, 2011).
In contrast to (Boumal and Absil, 2011), we set the term in the objective function based on the
first-order finite geometric differences to zero, such that the solutions of the regression problem
are approximating cubic splines.
}
\examples{
set.seed(2)
P <- rExamples(100, example = 'gaussian')
P.spline <- pdSplineReg(P$per, P$per, lam = 0.5, Nd = 25)
## Examine matrix-component (1,1)
plot((1:50)/50, Re(P$per[1, 1, ]), type = "l", lty = 2) ## noisy observations
lines((1:25)/25, Re(P.spline$f[1, 1, ])) ## penalized regression estimator
lines((1:50)/50, Re(P$f[1, 1, ]), col = 2, lty = 2) ## smooth target

}
\references{
Boumal, N. and Absil, P-A. (2011). A discrete regression method on manifolds and its applications
to data on SO(n). \emph{IFAC Proceedings Volumes}, 44, 2284-2289.
}
