% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sens.R
\name{sens}
\alias{sens}
\title{Sensitivity Analysis for Unobserved Confounding on Path-Specific Causal Effects}
\usage{
sens(
  object,
  confounded = "M1",
  estimand = "via M1",
  estimator = c("pure", "hybrid"),
  decomp = c("Type I", "Type II"),
  gamma_values = NULL,
  eta_values = NULL
)
}
\arguments{
\item{object}{a fitted model object returned by the \code{\link{paths}} function.}

\item{confounded}{a character string indicating the mediator whose relationship with the outcome
is potentially confounded. One of \{\code{"M1", "M2", \ldots}\}.}

\item{estimand}{a character string indicating the estimand whose sensitivity to unobserved
confounding is being investigated. One of \{\code{"M1", "M2", \ldots}\} or \code{"direct"}.}

\item{estimator}{type of estimator, the pure imputation estimator (\code{"pure"}) or the imputation-based
weighting estimator (\code{"hybrid"}).}

\item{decomp}{type of decomposition, \code{"Type I"} or \code{"Type II"}.}

\item{gamma_values}{potential values of the \eqn{\gamma} parameter, which denotes the average effect of
the unobserved confounder \eqn{U} on the outcome given pretreatment covariates \eqn{X}, treatment \eqn{A},
and mediators \eqn{M_1,\ldots, M_k}. If not provided, it is defaulted to a range of 20 values from
-\eqn{\textup{sd}(Y)} to \eqn{\textup{sd}(Y)}, where \eqn{sd} denotes standard deviation and \eqn{Y}
denotes the outcome variable.}

\item{eta_values}{potential values of the \eqn{\eta} parameter, which denotes the difference in the
prevalence of the unobserved confounder \eqn{U} between treated and untreated units given pretreatment
covariates \eqn{X} and mediators \eqn{M_1,\ldots, M_k}. If not provided, it is defaulted to a range of
20 values from -\eqn{sd(A)} to \eqn{sd(A)}, where \eqn{sd} denotes standard deviation and \eqn{A} denotes
the treatment variable.}
}
\value{
A list containing the following elements \describe{
  \item{original}{original estimate of the corresponding path-specific effect.}
  \item{adjusted}{a data frame where each row represents a potential combination of \eqn{\gamma} and \eqn{\eta},
  the corresponding bias, bias-adjusted estimate, and an indicator for whether the bias-adjusted estimate is
  of the opposite sign to the original estimate.}
  }
}
\description{
\code{sens} implements a set of bias formulas detailed in Zhou and Yamamoto (2020) for assessing
the sensitivity of estimated path-specific effects to an unobserved confounder \eqn{U} of a mediator-outcome
relationship. The user provides a fitted \code{paths} object, the mediator whose relationship
with the outcome is potentially confounded, the estimand whose sensitivity to unobserved
confounding is being investigated, type of estimator, type of decomposition, and possible values of
the \eqn{\gamma} and \eqn{\eta} parameters.
}
\examples{
data(tatar)

m1 <- c("trust_g1", "victim_g1", "fear_g1")
m2 <- c("trust_g2", "victim_g2", "fear_g2")
m3 <- c("trust_g3", "victim_g3", "fear_g3")
mediators <- list(m1, m2, m3)

formula_m0 <- annex ~ kulak + prosoviet_pre + religiosity_pre + land_pre +
  orchard_pre + animals_pre + carriage_pre + otherprop_pre + violence
formula_m1 <- update(formula_m0,    ~ . + trust_g1 + victim_g1 + fear_g1)
formula_m2 <- update(formula_m1,    ~ . + trust_g2 + victim_g2 + fear_g2)
formula_m3 <- update(formula_m2,    ~ . + trust_g3 + victim_g3 + fear_g3)
formula_ps <- violence ~ kulak + prosoviet_pre + religiosity_pre +
  land_pre + orchard_pre + animals_pre + carriage_pre + otherprop_pre

####################################################
# Causal Paths Analysis using GLM
####################################################

# outcome models
glm_m0 <- glm(formula_m0, family = binomial("logit"), data = tatar)
glm_m1 <- glm(formula_m1, family = binomial("logit"), data = tatar)
glm_m2 <- glm(formula_m2, family = binomial("logit"), data = tatar)
glm_m3 <- glm(formula_m3, family = binomial("logit"), data = tatar)
glm_ymodels <- list(glm_m0, glm_m1, glm_m2, glm_m3)

# propensity score model
glm_ps <- glm(formula_ps, family = binomial("logit"), data = tatar)

# causal paths analysis using glm
# note: For illustration purposes only a small number of bootstrap replicates are used
paths_glm <- paths(a = "violence", y = "annex", m = mediators,
  glm_ymodels, ps_model = glm_ps, data = tatar, nboot = 3)

# sensitivity analysis for the path-specific effect via M1
sens_glm <- sens(paths_glm, confounded = "M1", estimand = "via M1",
  gamma_values = - seq(0, 0.5, 0.005), eta_values = seq(-0.5, 0.5, 0.005))

plot(sens_glm)

}
\references{
Zhou, Xiang and Teppei Yamamoto. 2020. "\href{https://osf.io/2rx6p}{Tracing Causal Paths from Experimental and Observational Data}".
}
\seealso{
\code{\link{paths}}, \code{\link{plot.sens}}
}
