% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/import_data.R
\name{import_data}
\alias{import_data}
\title{Import data into 'GRASS.'}
\usage{
import_data(
  dem,
  band = 1,
  sites,
  streams = NULL,
  snap_streams = FALSE,
  pred_sites = NULL,
  predictor_raster = NULL,
  predictor_r_names = NULL,
  predictor_vector = NULL,
  predictor_v_names = NULL
)
}
\arguments{
\item{dem}{character; path to DEM (digital elevation model) raster file.}

\item{band}{integer (optional); defines which band of the dem to use}

\item{sites}{character string or object; path to sites vector file (ESRI shape) 
or sp or sf data object.}

\item{streams}{character string or object (optional); path to network vector 
file (ESRI shape) or sp or sf data object. If available this can be burnt into the DEM 
in \code{\link{derive_streams}}}

\item{snap_streams}{boolean (optional); snap line ends.
If TRUE line ends of the streams are snapped to the next feature if they are
 unconnected with threshold of 10 m using 'GRASS' function v.clean.}

\item{pred_sites}{character string vector or object(s) (optional); path(s) to prediction sites 
vector files (ESRI shape) or sp or sf data object. 
Different formats (i.e. path and objects) must not be mixed; more than one sf or sp
object must be provided as a list, not concatenated with \code{c}.}

\item{predictor_raster}{character vector (optional); paths to raster data to 
import as predictors.}

\item{predictor_r_names}{character string vector (optional); names for potential predictor
variables in raster format; if not provided \code{perdictor_raster} is used.}

\item{predictor_vector}{character string vector of object(s) (optional); path(s)
to vector data (ESRI shape) or sp or sf object names to import as predictors.
Different formats (i.e. path and objects) must not be mixed; more than one sf or sp
object must be provided as a list, not concatenated with \code{c}.}

\item{predictor_v_names}{character vector (optional); names for potential predictor
variables in vector format ; if not provided \code{perdictor_vector} is used.}
}
\value{
Nothing, the data is loaded into the 'GRASS' session (mapset PERMANENT).
The DEM is stored as raster 'dem', sites as vector 'sites_o', prediction sites
as vector using the original file names with an appended '_o' (without extension),
streams as vector 'streams_o' in the 'GRASS' location. Additionally, predictor 
raster map(s) can be read in and are stored in 'GRASS' using either the 
original file names (without extension) or using the names provides in 
\code{predictor_r_names}. The latter option may be useful if ArcGIS grid data 
(typically stored as 'grid_name/w001001.adf') are used. Likewise, predictor
vector maps can be read in from Esri Shape file (given as the full file path)
or as sf or sp objects. Potential predictor data can also be read in later, e.g.
using GRASS commands \href{https://grass.osgeo.org/grass78/manuals/v.import.html}{v.import} 
or \href{https://grass.osgeo.org/grass78/manuals/r.in.gdal.html}{r.in.gdal}
(see examples below).
}
\description{
This function loads a DEM (digital elevation model) and sites data (both required) into the 'GRASS' session.
Optionally, prediction sites and streams data can be loaded and the streams 
may be corrected by snapping to prevent lose ends. Likewise, potential predictor 
maps (raster or vector format) can be loaded.
}
\details{
All vector data (sites, streams and potential predictors) is imported 
into the current location using \href{https://grass.osgeo.org/grass78/manuals/v.import.html}{v.import}.
Hence, if the projections does not match to the one of the DEM (which was used
to specify the location in \code{\link{setup_grass_environment}}) the maps 
are projected and imported on the fly.
All raster data are not transformed but it is assumed that they have the same 
projection as the current location. Hence, it is important to make sure that 
they all have indeed the same projection (and same cell size) and that the correct
one is set in \code{\link{setup_grass_environment}}. If this condition is not met,
the raster data should be preprocessed before importing.
Use \code{\link{check_projection}} to compare the projection of a raster data set and
the one of the current location (i.e the one of the dem).
}
\note{
A GRASS session must be initiated and setup before, see \code{\link{setup_grass_environment}}.

If sites, pred_sites and / or streams are sp objects it is important that they 
have a datum defined otherwise the import will not work. Hence, it is e.g. 
better to use proj4string = CRS("+proj=tmerc +lat_0=0 +lon_0=9 +k=1 +x_0=3500000 +y_0=0 +datum=potsdam +units=m +no_defs")
instead of proj4string = CRS("+proj=tmerc +lat_0=0 +lon_0=9 +k=1 +x_0=3500000 +y_0=0 +ellps=bessel +towgs84=598.1,73.7,418.2,0.202,0.045,-2.455,6.7 +units=m +no_defs"))
when defining sp objects. However, please note that proj4 is outdated 
(\code{link{https://www.r-spatial.org/r/2020/03/17/wkt.html}}) and will at least rise warnings.
}
\examples{
\donttest{
dem_path <- system.file("extdata", "nc", "elev_ned_30m.tif", package = "openSTARS")
if(.Platform$OS.type == "windows"){
  grass_program_path = "c:/Program Files/GRASS GIS 7.6"
  } else {
  grass_program_path = "/usr/lib/grass78/"
  }

setup_grass_environment(dem = dem_path, 
                        gisBase = grass_program_path,      
                        remove_GISRC = TRUE,
                        override = TRUE
                        )
gmeta()

# Load files into GRASS
dem_path <- system.file("extdata", "nc", "elev_ned_30m.tif", package = "openSTARS")
sites_path <- system.file("extdata", "nc", "sites_nc.shp", package = "openSTARS")
streams_path <- system.file("extdata", "nc", "streams.shp", package = "openSTARS")
preds_v_path <- system.file("extdata", "nc", "pointsources.shp", package = "openSTARS")
preds_r_path <- system.file("extdata", "nc", "landuse_r.tif", package = "openSTARS")
                 
import_data(dem = dem_path, sites = sites_path, streams = streams_path,
            predictor_vector = preds_v_path, predictor_raster = preds_r_path)

# Plot data
library(sp)
dem <- readRAST("dem", ignore.stderr = TRUE, plugin = FALSE)
sites_orig <-  readVECT("sites_o", ignore.stderr = TRUE)
lu <- readRAST("landuse_r", ignore.stderr = TRUE, plugin = FALSE)
# import additional vector data
fp <-  system.file("extdata", "nc", "pointsources.shp", package = "openSTARS")
execGRASS("v.import", flags = c("overwrite", "quiet"),
parameters = list(
  input = fp,
  output =  "psources",
  extent = "region"),  # to import into current regien
  intern = TRUE, ignore.stderr = TRUE)
  
#plot(dem, col = terrain.colors(20))
plot(dem, col = grey.colors(20))
points(sites_orig, pch = 4)
ps <- readVECT("psources")
points(ps, bg = "red", pch = 21, col = "grey", cex = 1.5)

# plot landuse data
library(raster)
par(mfcol = c(1,1), mar = c(5,4,4,2))
op <- par()
par(xpd = FALSE)
plot(raster(lu), legend = FALSE, xaxt = "n", yaxt = "n", bty = "n",
     col = c("red", "goldenrod", "green", "forestgreen","darkgreen", "blue", "lightblue"))
par(xpd = TRUE)
legend("bottom", cex = 0.75,
       legend = c("developed", "agriculture", "herbaceous", 
                  "shrubland", "forest", "water", "sediment"),
       fill = c("red", "goldenrod", "green", 
                "forestgreen","darkgreen", "blue", "lightblue"),
       horiz = TRUE, inset = -0.175)
par <- op
}
}
\author{
Eduard Szoecs, \email{eduardszoecs@gmial.com},  Mira Kattwinkel
 \email{mira.kattwinkel@gmx.net}
}
