% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sparseODPC.R
\name{crit.sparse_odpc}
\alias{crit.sparse_odpc}
\title{Automatic Choice of Regularization Parameters for Sparse One-Sided Dynamic Principal Components using a BIC type criterion}
\usage{
crit.sparse_odpc(
  Z,
  k_list = 1:3,
  max_num_comp = 1,
  nlambda = 20,
  tol = 1e-04,
  niter_max = 500,
  eps = 0.001,
  ncores = 1
)
}
\arguments{
\item{Z}{Data matrix. Each column is a different time series.}

\item{k_list}{List of values of k to choose from.}

\item{max_num_comp}{Maximum possible number of components to compute.}

\item{nlambda}{Length of penalty sequence.}

\item{tol}{Relative precision. Default is 1e-4.}

\item{niter_max}{Integer. Maximum number of iterations. Default is 500.}

\item{eps}{Between 0 and 1, used to build penalty sequence}

\item{ncores}{Number of cores to use in parallel computations}
}
\value{
An object of class odpcs, that is, a list of length equal to the number of computed components, each computed using the optimal value of k. 
The i-th entry of this list is an object of class \code{odpc}, that is, a list with entries
\item{f}{Coordinates of the i-th dynamic principal component corresponding to the periods \eqn{k_1 + 1,\dots,T}.}
\item{mse}{Mean squared error of the reconstruction using the first i components.}
\item{k1}{Number of lags used to define the i-th dynamic principal component f.}
\item{k2}{Number of lags of f used to reconstruct.}
\item{alpha}{Vector of intercepts corresponding to f.}
\item{a}{Vector that defines the i-th dynamic principal component}
\item{B}{Matrix of loadings corresponding to f. Row number \eqn{k} is the vector of \eqn{k-1} lag loadings.}
\item{call}{The matched call.}
\item{conv}{Logical. Did the iterations converge?}
\item{lambda}{Regularization parameter used for this component}
\code{components}, \code{fitted}, \code{plot} and \code{print} methods are available for this class.
}
\description{
Computes Sparse One-Sided Dynamic Principal Components, choosing the number of components and regularization parameters automatically, using a BIC type criterion.
}
\details{
First \code{\link{crit.odpc}} is called to choose the number of lags and of components to use. Each component is then computed using a regularized version of the
odpc objective function (see \code{\link{odpc}}), where the L1 norm of the \eqn{\mathbf{a}} vector is penalized. The penalization parameter \eqn{\lambda} is chosen from a grid of candidates
of size \code{nlambda}, seeking to minimize the following BIC type criterion
\deqn{
\log(MSE(\mathbf{a}_{\lambda},\mathbf{\alpha}_{\lambda}, \mathbf{B}_{\lambda} )) + \frac{\log(T^{\ast} m)}{T^{\ast}m} \Vert \mathbf{a}_{\lambda}\Vert_{0},
}
where \eqn{\mathbf{a}_{\lambda},\mathbf{B}_{\lambda} } are the estimates associated with a given \eqn{\lambda}, \eqn{m} is the number of series and
\eqn{T^{\ast}} is the number of periods being reconstructed.
}
\examples{
T <- 50 #length of series
m <- 10 #number of series
set.seed(1234)
f <- rnorm(T + 1)
x <- matrix(0, T, m)
u <- matrix(rnorm(T * m), T, m)
for (i in 1:m) {
  x[, i] <- 10 * sin(2 * pi * (i/m)) * f[1:T] + 10 * cos(2 * pi * (i/m)) * f[2:(T + 1)] + u[, i]
}
fit <- crit.sparse_odpc(x, k_list = 1, ncores = 1)
}
\references{
Peña D., Smucler E. and Yohai V.J. (2017). “Forecasting Multiple Time Series with One-Sided Dynamic Principal Components.” Available at https://arxiv.org/abs/1708.04705.
}
\seealso{
\code{\link{odpc}}, \code{\link{crit.odpc}}, \code{\link{forecast.odpcs}}
}
