
#' Simulated Annealing call simulations function
#'
#' @description Simulated Annealing call simulations function
#'
#' @param nl nl object
#' @param seed current model seed
#' @param cleanup.csv TRUE/FALSE, if TRUE temporary created csv output files will be deleted after gathering results.
#' @param cleanup.xml TRUE/FALSE, if TRUE temporary created xml output files will be deleted after gathering results.
#' @param cleanup.bat TRUE/FALSE, if TRUE temporary created bat/sh output files will be deleted after gathering results.
#' @param silent TRUE/FALSE, if FALSE prints the current seed and siminputrow after successful execution of a simulation (only for sequential execution).
#' @aliases util_run_nl_dyn_GenSA
#' @rdname util_run_nl_dyn_GenSA
#' @keywords internal
util_run_nl_dyn_GenSA <- function(nl,
                                  seed,
                                  cleanup.csv,
                                  cleanup.xml,
                                  cleanup.bat,
                                  silent) {

  # Get GenSA object from simdesign:
  gensa <- getsim(nl, "simobject")

  # Call the GenSA function from the GenSA package:
  results <- GenSA::GenSA(
    par = NULL,
    fn = function(par, ...) {
      util_run_nl_dyn_GenSA_fn(
        param = par,
        nl = nl,
        evalcrit = gensa$evalcrit,
        seed = seed,
        cleanup.csv = cleanup.csv,
        cleanup.xml = cleanup.xml,
        cleanup.bat = cleanup.bat,
        silent = silent,
        ...
      )
    },
    control = gensa$control,
    lower = as.vector(gensa$lower),
    upper = as.vector(gensa$upper)
  )

  return(results)
}


#' Simulated Annealing run simulation function
#'
#' @description Simulated Annealing run simulation function
#'
#' @param param vector of model parameters, generated by GenSA function
#' @param nl nl object
#' @param evalcrit evaluation criterion for simulated annealing
#' @param seed current model seed
#' @param cleanup.csv TRUE/FALSE, if TRUE temporary created csv output files will be deleted after gathering results.
#' @param cleanup.xml TRUE/FALSE, if TRUE temporary created xml output files will be deleted after gathering results.
#' @param cleanup.bat TRUE/FALSE, if TRUE temporary created bat/sh output files will be deleted after gathering results.
#' @param silent TRUE/FALSE, if FALSE prints the current seed and siminputrow after successful execution of a simulation (only for sequential execution).
#' @aliases util_run_nl_dyn_GenSA_fn
#' @rdname util_run_nl_dyn_GenSA_fn
#' @keywords internal
util_run_nl_dyn_GenSA_fn <- function(param,
                                     nl,
                                     evalcrit,
                                     seed,
                                     cleanup.csv,
                                     cleanup.xml,
                                     cleanup.bat,
                                     silent) {

  # Generate a parameterset:
  names(param) <- names(getexp(nl, "variables"))

  ## Generate parameterset
  gensa_param <- tibble::as.tibble(t(param))

  ## Add constants if any:
  if(length(getexp(nl, "constants")) > 0)
  {
    gensa_param <- tibble::as.tibble(cbind(gensa_param,
                                           getexp(nl, "constants"),
                                           stringsAsFactors = FALSE))

  }

  # Attach current parameterisation to nl object:
  setsim(nl, "siminput") <- gensa_param
  # Call netlogo:
  results <- run_nl_one(
    nl = nl,
    siminputrow = 1,
    seed = seed,
    cleanup.csv = cleanup.csv,
    cleanup.xml = cleanup.xml,
    cleanup.bat = cleanup.bat,
    silent = silent
  )

  # Select metric for gensa:
  results <- results[[evalcrit]]
  # Calc mean and convert to numeric:
  if (length(results) > 1) {
    results <- mean(results)
  }
  results <- as.numeric(results)

  return(results)
}


#' Genetic Algorithm call simulations function
#'
#' @description Genetic Algorithm call simulations function
#'
#' @param nl nl object
#' @param seed current model seed
#' @param cleanup.csv TRUE/FALSE, if TRUE temporary created csv output files will be deleted after gathering results.
#' @param cleanup.xml TRUE/FALSE, if TRUE temporary created xml output files will be deleted after gathering results.
#' @param cleanup.bat TRUE/FALSE, if TRUE temporary created bat/sh output files will be deleted after gathering results.
#' @param silent TRUE/FALSE, if FALSE prints the current seed and siminputrow after successful execution of a simulation (only for sequential execution).
#' @aliases util_run_nl_dyn_GenAlg
#' @rdname util_run_nl_dyn_GenAlg
#' @keywords internal
util_run_nl_dyn_GenAlg <- function(nl,
                                   seed,
                                   cleanup.csv,
                                   cleanup.xml,
                                   cleanup.bat,
                                   silent) {

  # Get GenSA object from simdesign:
  galg <- getsim(nl, "simobject")

  # Call the GenSA function from the GenSA package:
  results <- genalg::rbga(
    stringMin = galg$lower,
    stringMax = galg$upper,
    popSize = galg$popSize,
    iters = galg$iters,
    elitism = galg$elitism,
    mutationChance = galg$mutationChance,
    evalFunc = function(par, ...) {
      util_run_nl_dyn_GenAlg_fn(
        param = par,
        nl = nl,
        evalcrit = galg$evalcrit,
        seed = seed,
        cleanup.csv = cleanup.csv,
        cleanup.xml = cleanup.xml,
        cleanup.bat = cleanup.bat,
        silent = silent,
        ...
      )
    }
  )

  return(results)
}


#' Genetic Algorithm run simulation function
#'
#' @description Genetic Algorithm run simulation function
#'
#' @param param vector of model parameters, generated by GenSA function
#' @param nl nl object
#' @param evalcrit evaluation criterion for simulated annealing
#' @param seed current model seed
#' @param cleanup.csv TRUE/FALSE, if TRUE temporary created csv output files will be deleted after gathering results.
#' @param cleanup.xml TRUE/FALSE, if TRUE temporary created xml output files will be deleted after gathering results.
#' @param cleanup.bat TRUE/FALSE, if TRUE temporary created bat/sh output files will be deleted after gathering results.
#' @param silent TRUE/FALSE, if FALSE prints the current seed and siminputrow after successful execution of a simulation (only for sequential execution).
#' @aliases util_run_nl_dyn_GenAlg_fn
#' @rdname util_run_nl_dyn_GenAlg_fn
#' @keywords internal
util_run_nl_dyn_GenAlg_fn <- function(param,
                                      nl,
                                      evalcrit,
                                      seed,
                                      cleanup.csv,
                                      cleanup.xml,
                                      cleanup.bat,
                                      silent) {

  # Generate a parameterset:
  names(param) <- names(getexp(nl, "variables"))

  ## Generate parameterset
  gensa_param <- tibble::as.tibble(t(param))

  ## Add constants if any:
  if(length(getexp(nl, "constants")) > 0)
  {
    gensa_param <- tibble::as.tibble(cbind(gensa_param,
                                           getexp(nl, "constants"),
                                           stringsAsFactors = FALSE))

  }

  # Attach current parameterisation to nl object:
  setsim(nl, "siminput") <- gensa_param
  # Call netlogo:
  results <- run_nl_one(
    nl = nl,
    siminputrow = 1,
    seed = seed,
    cleanup.csv = cleanup.csv,
    cleanup.xml = cleanup.xml,
    cleanup.bat = cleanup.bat,
    silent = silent
  )

  # Select metric for gensa:
  results <- results[[evalcrit]]
  # Calc mean and convert to numeric:
  if (length(results) > 1) {
    results <- mean(results)
  }
  results <- as.numeric(results)

  return(results)
}
