% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rmixn.R
\name{rmixn}
\alias{rmixn}
\title{Fast simulation of r.v.s from a mixture of multivariate normal densities}
\usage{
rmixn(n, mu, sigma, w, ncores = 1, isChol = FALSE, retInd = FALSE,
  A = NULL)
}
\arguments{
\item{n}{number of random vectors to be simulated.}

\item{mu}{an (m x d) matrix, where m is the number of mixture components.}

\item{sigma}{as list of m covariance matrices (d x d) on for each mixture component. 
Alternatively it can be a list of m cholesky decomposition of the covariance. 
In that case \code{isChol} should be set to \code{TRUE}.}

\item{w}{vector of length m, containing the weights of the mixture components.}

\item{ncores}{Number of cores used. The parallelization will take place only if OpenMP is supported.}

\item{isChol}{boolean set to true is \code{sigma} is the cholesky decomposition of the covariance matrix.}

\item{retInd}{when set to \code{TRUE} an attribute called "index" will be added to the output matrix of random variables.
This is a vector specifying to which mixture components each random vector belongs. \code{FALSE} by default.}

\item{A}{an (optional) numeric matrix of dimension (n x d), which will be used to store the output random variables.
It is useful when n and d are large and one wants to call \code{rmvn()} several times, without reallocating memory
for the whole matrix each time. NB: the element of \code{A} must be of class "numeric".}
}
\value{
If \code{A==NULL} (default) the output is an (n x d) matrix where the i-th row is the i-th simulated vector.
        If \code{A!=NULL} then the random vector are store in \code{A}, which is provided by the user, and the function
        returns \code{NULL}. Notice that if \code{retInd==TRUE} an attribute called "index" will be added to A.
        This is a vector specifying to which mixture components each random vector belongs.
}
\description{
Fast simulation of r.v.s from a mixture of multivariate normal densities
}
\details{
Notice that this function does not use one of the Random Number Generators (RNGs) provided by R, but one 
         of the parallel cryptographic RNGs described in (Salmon et al., 2011). It is important to point out that this
         RNG can safely be used in parallel, without risk of collisions between parallel sequence of random numbers.
         The initialization of the RNG depends on R's seed, hence the \code{set.seed()} function can be used to 
         obtain reproducible results. Notice though that changing \code{ncores} causes most of the generated numbers
         to be different even if R's seed is the same (see example below). NB: at the moment the RNG does not work
         properly on Solaris OS when \code{ncores>1}. Hence, \code{rmixn()} checks if the OS is Solaris and, if this the case, 
         it imposes \code{ncores==1}.
}
\examples{
# Create mixture of two components
mu <- matrix(rep(c(1, 2, 10, 20), 2), 2, 2, byrow = TRUE)
sigma <- list(diag(c(1, 10)), matrix(c(1, -0.9, -0.9, 1), 2, 2))
w <- c(0.1, 0.9)

# Simulate
X <- rmixn(1e4, mu, sigma, w, retInd = TRUE)
plot(X, pch = '.', col = attr(X, "index"))

# Simulate with fixed seed
set.seed(414)
rmixn(4, mu, sigma, w)

set.seed(414)
rmixn(4, mu, sigma, w)

set.seed(414)  
rmixn(4, mu, sigma, w, ncores = 2) # r.v. generated on the second core are different

###### Here we create the matrix that will hold the simulated random variables upfront.
A <- matrix(NA, 4, 2)
class(A) <- "numeric" # This is important. We need the elements of A to be of class "numeric". 

set.seed(414)
rmixn(4, mu, sigma, w, ncores = 2, A = A) # This returns NULL ...
A                                         # ... but the result is here

}
\references{
John K. Salmon, Mark A. Moraes, Ron O. Dror, and David E. Shaw (2011). Parallel Random Numbers: As Easy as 1, 2, 3.
             D. E. Shaw Research, New York, NY 10036, USA.
}
\author{
Matteo Fasiolo <matteo.fasiolo@gmail.com>, C++ RNG engine by Thijs van den Berg <http://sitmo.com/>.
}
