% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jsdgam.R
\name{jsdgam}
\alias{jsdgam}
\title{Fit Joint Species Distribution Models in \pkg{mvgam}}
\usage{
jsdgam(
  formula,
  factor_formula = ~-1,
  knots,
  factor_knots,
  data,
  newdata,
  family = poisson(),
  unit = time,
  species = series,
  share_obs_params = FALSE,
  priors,
  n_lv = 2,
  backend = getOption("brms.backend", "cmdstanr"),
  algorithm = getOption("brms.algorithm", "sampling"),
  control = list(max_treedepth = 10, adapt_delta = 0.8),
  chains = 4,
  burnin = 500,
  samples = 500,
  thin = 1,
  parallel = TRUE,
  threads = 1,
  silent = 1,
  run_model = TRUE,
  return_model_data = FALSE,
  ...
)
}
\arguments{
\item{formula}{A \code{formula} object specifying the GAM observation model formula. These are exactly like the formula
for a GLM except that smooth terms, \code{s()}, \code{te()}, \code{ti()}, \code{t2()}, as well as time-varying
\code{dynamic()} terms, nonparametric \code{gp()} terms and offsets using \code{offset()}, can be added to the right hand side
to specify that the linear predictor depends on smooth functions of predictors
(or linear functionals of these). Details of the formula syntax used by \pkg{mvgam}
can be found in \code{\link{mvgam_formulae}}}

\item{factor_formula}{A \code{formula} object specifying the linear predictor
effects for the latent factors. Use \code{by = trend} within calls to functional terms
(i.e. \code{s()}, \code{te()}, \code{ti()}, \code{t2()}, \code{dynamic()}, or \code{gp()}) to ensure that each factor
captures a different axis of variation. See the example below as an illustration}

\item{knots}{An optional \code{list} containing user specified knot values to be used for basis construction.
For most bases the user simply supplies the knots to be used, which must match up with the \code{k} value supplied
(note that the number of knots is not always just \code{k}). Different terms can use different numbers of knots,
unless they share a covariate}

\item{factor_knots}{An optional \code{list} containing user specified knot values to
be used for basis construction of any smooth terms in \code{factor_formula}.
For most bases the user simply supplies the knots to be used, which must match up with the \code{k} value supplied
(note that the number of knots is not always just \code{k}). Different terms can use different numbers of knots,
unless they share a covariate}

\item{data}{A \code{dataframe} or \code{list} containing the model response variable and covariates
required by the GAM \code{formula} and \code{factor_formula} objects}

\item{newdata}{Optional \code{dataframe} or \code{list} of test data containing the same variables
as in \code{data}. If included, the
observations in variable \code{y} will be set to \code{NA} when fitting the model so that posterior
simulations can be obtained}

\item{family}{\code{family} specifying the observation family for the outcomes. Currently supported
families are:
\itemize{
\item\code{gaussian()} for real-valued data
\item\code{betar()} for proportional data on \verb{(0,1)}
\item\code{lognormal()} for non-negative real-valued data
\item\code{student_t()} for real-valued data
\item\code{Gamma()} for non-negative real-valued data
\item\code{bernoulli()} for binary data
\item\code{poisson()} for count data
\item\code{nb()} for overdispersed count data
\item\code{binomial()} for count data with imperfect detection when the number of trials is known;
note that the \code{cbind()} function must be used to bind the discrete observations and the discrete number
of trials
\item\code{beta_binomial()} as for \code{binomial()} but allows for overdispersion}
Default is \code{poisson()}. See \code{\link{mvgam_families}} for more details}

\item{unit}{The unquoted name of the variable that represents the unit of analysis in \code{data} over
which latent residuals should be correlated. This variable should be either a
\code{numeric} or \code{integer} variable in the supplied \code{data}.
Defaults to \code{time} to be consistent with other functionalities
in \pkg{mvgam}, though note that the data need not be time series in this case. See examples below
for further details and explanations}

\item{species}{The unquoted name of the \code{factor} variable that indexes
the different response units in \code{data} (usually \code{'species'} in a JSDM).
Defaults to \code{series} to be consistent with other \code{mvgam} models}

\item{share_obs_params}{\code{logical}. If \code{TRUE} and the \code{family}
has additional family-specific observation parameters (e.g. variance components in
\code{student_t()} or \code{gaussian()}, or dispersion parameters in \code{nb()} or \code{betar()}),
these parameters will be shared across all outcome variables. This is handy if you have multiple
outcomes (time series in most \code{mvgam} models) that you believe share some properties,
such as being from the same species over different spatial units. Default is \code{FALSE}.}

\item{priors}{An optional \code{data.frame} with prior
definitions (in Stan syntax) or, preferentially, a vector containing
objects of class \code{brmsprior} (see. \code{\link[brms]{prior}} for details).
See \link{get_mvgam_priors} and for more information on changing default prior distributions}

\item{n_lv}{\code{integer} the number of latent factors to use for modelling
residual associations.
Cannot be \verb{> n_species}. Defaults arbitrarily to \code{2}}

\item{backend}{Character string naming the package to use as the backend for fitting
the Stan model. Options are "cmdstanr" (the default) or "rstan". Can be set globally
for the current R session via the \code{"brms.backend"} option (see \code{\link{options}}). Details on
the rstan and cmdstanr packages are available at https://mc-stan.org/rstan/ and
https://mc-stan.org/cmdstanr/, respectively}

\item{algorithm}{Character string naming the estimation approach to use.
Options are \code{"sampling"} for MCMC (the default), \code{"meanfield"} for
variational inference with factorized normal distributions,
\code{"fullrank"} for variational inference with a multivariate normal
distribution, \code{"laplace"} for a Laplace approximation (only available
when using cmdstanr as the backend) or \code{"pathfinder"} for the pathfinder
algorithm (only currently available when using cmdstanr as the backend).
Can be set globally for the current \R session via the
\code{"brms.algorithm"} option (see \code{\link{options}}). Limited testing
suggests that \code{"meanfield"} performs best out of the non-MCMC approximations for
dynamic GAMs, possibly because of the difficulties estimating covariances among the
many spline parameters and latent trend parameters. But rigorous testing has not
been carried out}

\item{control}{A named \code{list} for controlling the sampler's behaviour. Valid elements include
\code{max_treedepth}, \code{adapt_delta} and \code{init}}

\item{chains}{\code{integer} specifying the number of parallel chains for the model. Ignored
if \code{algorithm \%in\% c('meanfield', 'fullrank', 'pathfinder', 'laplace')}}

\item{burnin}{\code{integer} specifying the number of warmup iterations of the Markov chain to run
to tune sampling algorithms. Ignored
if \code{algorithm \%in\% c('meanfield', 'fullrank', 'pathfinder', 'laplace')}}

\item{samples}{\code{integer} specifying the number of post-warmup iterations of the Markov chain to run for
sampling the posterior distribution}

\item{thin}{Thinning interval for monitors.  Ignored
if \code{algorithm \%in\% c('meanfield', 'fullrank', 'pathfinder', 'laplace')}}

\item{parallel}{\code{logical} specifying whether multiple cores should be used for
generating MCMC simulations in parallel. If \code{TRUE}, the number of cores to use will be
\code{min(c(chains, parallel::detectCores() - 1))}}

\item{threads}{\code{integer} Experimental option to use multithreading for within-chain
parallelisation in \code{Stan}. We recommend its use only if you are experienced with
\code{Stan}'s \code{reduce_sum} function and have a slow running model that cannot be sped
up by any other means. Currently works for all families when using \code{Cmdstan}
as the backend}

\item{silent}{Verbosity level between \code{0} and \code{2}. If \code{1} (the default), most of the informational
messages of compiler and sampler are suppressed. If \code{2}, even more messages are suppressed. The
actual sampling progress is still printed. Set \code{refresh = 0} to turn this off as well. If using
\code{backend = "rstan"} you can also set open_progress = FALSE to prevent opening additional
progress bars.}

\item{run_model}{\code{logical}. If \code{FALSE}, the model is not fitted but instead the function will
return the model file and the data / initial values that are needed to fit the model outside of \code{mvgam}}

\item{return_model_data}{\code{logical}. If \code{TRUE}, the list of data that is needed to fit the
model is returned, along with the initial values for smooth and AR parameters, once the model is fitted.
This will be helpful if users wish to modify the model file to add
other stochastic elements that are not currently available in \code{mvgam}.
Default is \code{FALSE} to reduce
the size of the returned object, unless \code{run_model == FALSE}}

\item{...}{Other arguments to pass to \link{mvgam}}
}
\value{
A \code{list} object of class \code{mvgam} containing model output,
the text representation of the model file,
the mgcv model output (for easily generating simulations at
unsampled covariate values), Dunn-Smyth residuals for each species and key information needed
for other functions in the package. See \code{\link{mvgam-class}} for details.
Use \code{methods(class = "mvgam")} for an overview on available methods
}
\description{
This function sets up a Joint Species Distribution Model whereby the residual associations among
species can be modelled in a reduced-rank format using a set of latent factors. The factor
specification is extremely flexible, allowing users to include spatial, temporal or any other type
of predictor effects to more efficiently capture unmodelled residual associations, while the
observation model can also be highly flexible (including all smooth, GP and other effects that
\pkg{mvgam} can handle)
}
\details{
Joint Species Distribution Models allow for responses of multiple species to be
learned hierarchically, whereby responses to environmental variables in \code{formula} can be partially
pooled and any latent, unmodelled residual associations can also be learned. In \pkg{mvgam}, both of
these effects can be modelled with the full power of latent factor Hierarchical GAMs, providing unmatched
flexibility to model full communities of species. When calling \link{jsdgam}, an initial State-Space model using
\code{trend = 'None'} is set up and then modified to include the latent factors and their linear predictors.
Consequently, you can inspect priors for these models using \link{get_mvgam_priors} by supplying the relevant
\code{formula}, \code{factor_formula}, \code{data} and \code{family} arguments and keeping the default \code{trend = 'None'}.

In a JSDGAM, the expectation of response \eqn{Y_{ij}} is modelled with

\deqn{g(\mu_{ij}) = X_i\beta + u_i\theta_j,}

where \eqn{g(.)} is a known link function,
\eqn{X} is a design matrix of linear predictors (with associated \eqn{\beta} coefficients),
\eqn{u} are \eqn{n_{lv}}-variate latent factors
(\eqn{n_{lv}}<<\eqn{n_{species}}) and
\eqn{\theta_j} are species-specific loadings on the latent factors, respectively. The design matrix
\eqn{X} and \eqn{\beta} coefficients are constructed and modelled using \code{formula} and can contain
any of \code{mvgam}'s predictor effects, including random intercepts and slopes, multidimensional penalized
smooths, GP effects etc... The factor loadings \eqn{\theta_j} are constrained for identifiability but can
be used to reconstruct an estimate of the species' residual variance-covariance matrix
using \eqn{\Theta \Theta'} (see the example below and \code{\link[=residual_cor]{residual_cor()}} for details).
The latent factors are further modelled using:
\deqn{
u_i \sim \text{Normal}(Q_i\beta_{factor}, 1) \quad
}
where the second design matrix \eqn{Q} and associated \eqn{\beta_{factor}} coefficients are
constructed and modelled using \code{factor_formula}. Again, the effects that make up this linear
predictor can contain any of \code{mvgam}'s allowed predictor effects, providing enormous flexibility for
modelling species' communities.
}
\examples{
\donttest{
# Simulate latent count data for 500 spatial locations and 10 species
set.seed(0)
N_points <- 500
N_species <- 10

# Species-level intercepts (on the log scale)
alphas <- runif(N_species, 2, 2.25)

# Simulate a covariate and species-level responses to it
temperature <- rnorm(N_points)
betas <- runif(N_species, -0.5, 0.5)

# Simulate points uniformly over a space
lon <- runif(N_points, min = 150, max = 155)
lat <- runif(N_points, min = -20, max = -19)

# Set up spatial basis functions as a tensor product of lat and lon
sm <- mgcv::smoothCon(mgcv::te(lon, lat, k = 5),
                      data = data.frame(lon, lat),
                      knots = NULL)[[1]]

# The design matrix for this smooth is in the 'X' slot
des_mat <- sm$X
dim(des_mat)

# Function to generate a random covariance matrix where all variables
# have unit variance (i.e. diagonals are all 1)
random_Sigma = function(N){
  L_Omega <- matrix(0, N, N);
  L_Omega[1, 1] <- 1;
  for (i in 2 : N) {
    bound <- 1;
    for (j in 1 : (i - 1)) {
      L_Omega[i, j] <- runif(1, -sqrt(bound), sqrt(bound));
      bound <- bound - L_Omega[i, j] ^ 2;
    }
    L_Omega[i, i] <- sqrt(bound);
  }
  Sigma <- L_Omega \%*\% t(L_Omega);
  return(Sigma)
}

# Simulate a variance-covariance matrix for the correlations among
# basis coefficients
Sigma <- random_Sigma(N = NCOL(des_mat))

# Now simulate the species-level basis coefficients hierarchically, where
# spatial basis function correlations are a convex sum of a base correlation
# matrix and a species-level correlation matrix
basis_coefs <- matrix(NA, nrow = N_species, ncol = NCOL(Sigma))
base_field <- mgcv::rmvn(1, mu = rep(0, NCOL(Sigma)), V = Sigma)
for(t in 1:N_species){
  corOmega <- (cov2cor(Sigma) * 0.7) +
                 (0.3 * cov2cor(random_Sigma(N = NCOL(des_mat))))
  basis_coefs[t, ] <- mgcv::rmvn(1, mu = rep(0, NCOL(Sigma)), V = corOmega)
}

# Simulate the latent spatial processes
st_process <- do.call(rbind, lapply(seq_len(N_species), function(t){
  data.frame(lat = lat,
             lon = lon,
             species = paste0('species_', t),
             temperature = temperature,
             process = alphas[t] +
               betas[t] * temperature +
               des_mat \%*\% basis_coefs[t,])
}))

# Now take noisy observations at some of the points (60)
obs_points <- sample(1:N_points, size = 60, replace = FALSE)
obs_points <- data.frame(lat = lat[obs_points],
                         lon = lon[obs_points],
                         site = 1:60)

# Keep only the process data at these points
st_process \%>\%
  dplyr::inner_join(obs_points, by = c('lat', 'lon')) \%>\%
  # now take noisy Poisson observations of the process
  dplyr::mutate(count = rpois(NROW(.), lambda = exp(process))) \%>\%
  dplyr::mutate(species = factor(species,
                                 levels = paste0('species_', 1:N_species))) \%>\%
  dplyr::group_by(lat, lon) -> dat

# View the count distributions for each species
library(ggplot2)
ggplot(dat, aes(x = count)) +
  geom_histogram() +
  facet_wrap(~ species, scales = 'free')

ggplot(dat, aes(x = lon, y = lat, col = log(count + 1))) +
  geom_point(size = 2.25) +
  facet_wrap(~ species, scales = 'free') +
  scale_color_viridis_c() +
  theme_classic()

# Inspect default priors for a joint species model with three spatial factors
priors <- get_mvgam_priors(formula = count ~
                            # Environmental model includes random slopes for
                            # a linear effect of temperature
                            s(species, bs = 're', by = temperature),

                          # Each factor estimates a different nonlinear spatial process, using
                          # 'by = trend' as in other mvgam State-Space models
                          factor_formula = ~ gp(lon, lat, k = 6, by = trend) - 1,
                          n_lv = 3,

                          # The data and grouping variables
                          data = dat,
                          unit = site,
                          species = species,

                          # Poisson observations
                          family = poisson())
head(priors)

# Fit a JSDM that estimates hierarchical temperature responses
# and that uses three latent spatial factors
mod <- jsdgam(formula = count ~
                # Environmental model includes random slopes for a
                # linear effect of temperature
                s(species, bs = 're', by = temperature),

              # Each factor estimates a different nonlinear spatial process, using
              # 'by = trend' as in other mvgam State-Space models
              factor_formula = ~ gp(lon, lat, k = 6, by = trend) - 1,
              n_lv = 3,

              # Change default priors for fixed random effect variances and
              # factor P marginal deviations to standard normal
              priors = c(prior(std_normal(),
                               class = sigma_raw),
                         prior(std_normal(),
                               class = `alpha_gp_trend(lon, lat):trendtrend1`),
                         prior(std_normal(),
                               class = `alpha_gp_trend(lon, lat):trendtrend2`),
                         prior(std_normal(),
                               class = `alpha_gp_trend(lon, lat):trendtrend3`)),

              # The data and the grouping variables
              data = dat,
              unit = site,
              species = species,

              # Poisson observations
              family = poisson(),
              chains = 2,
              silent = 2)

# Plot species-level intercept estimates
plot_predictions(mod, condition = 'species',
                 type = 'link')

# Plot species' hierarchical responses to temperature
plot_predictions(mod, condition = c('temperature', 'species', 'species'),
                 type = 'link')

# Plot posterior median estimates of the latent spatial factors
plot(mod, type = 'smooths', trend_effects = TRUE)

# Or using gratia, if you have it installed
if(requireNamespace('gratia', quietly = TRUE)){
  gratia::draw(mod, trend_effects = TRUE)
}

# Calculate residual spatial correlations
post_cors <- residual_cor(mod)
names(post_cors)
# Look at lower and upper credible interval estimates for
# some of the estimated correlations
post_cors$cor[1:5, 1:5]
post_cors$cor_upper[1:5, 1:5]
post_cors$cor_lower[1:5, 1:5]
# A quick and dirty plot of the posterior median correlations
image(post_cors$cor)

# Posterior predictive checks and ELPD-LOO can ascertain model fit
pp_check(mod, type = "pit_ecdf_grouped",
         group = "species", ndraws = 100)
loo(mod)

# Forecast log(counts) for entire region (site value doesn't matter as long
# as each spatial location has a different and unique site identifier);
# note this calculation takes a few minutes because of the need to calculate
# draws from the stochastic latent factors
newdata <- st_process \%>\%
                   dplyr::mutate(species = factor(species,
                                                  levels = paste0('species_',
                                                                  1:N_species))) \%>\%
                   dplyr::group_by(lat, lon) \%>\%
                   dplyr::mutate(site = dplyr::cur_group_id()) \%>\%
                   dplyr::ungroup()
preds <- predict(mod, newdata = newdata)

# Plot the median log(count) predictions on a grid
newdata$log_count <- preds[,1]
ggplot(newdata, aes(x = lon, y = lat, col = log_count)) +
  geom_point(size = 1.5) +
  facet_wrap(~ species, scales = 'free') +
  scale_color_viridis_c() +
  theme_classic()
}
}
\references{
Nicholas J Clark & Konstans Wells (2020). Dynamic generalised additive models (DGAMs) for forecasting discrete ecological time series.
Methods in Ecology and Evolution. 14:3, 771-784.
\cr
\cr
David I Warton, F Guillaume Blanchet, Robert B O’Hara, Otso Ovaskainen, Sara Taskinen, Steven C
Walker & Francis KC Hui (2015). So many variables: joint modeling in community ecology.
Trends in Ecology & Evolution 30:12, 766-779.
}
\seealso{
\code{\link[=mvgam]{mvgam()}}, \code{\link[=residual_cor]{residual_cor()}}
}
\author{
Nicholas J Clark
}
