% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/observation_plot.R
\name{observation_plot}
\alias{observation_plot}
\title{SHAP Observation Plot}
\usage{
observation_plot(
  variable_values,
  shap_values,
  expected_value,
  names = NULL,
  num_vars = 10,
  fill_colors = c("#A54657", "#0D3B66"),
  connect_color = "#849698",
  expected_color = "#849698",
  predicted_color = "#EE964B",
  title = "Individual Observation Explanation",
  font_family = "Times New Roman"
)
}
\arguments{
\item{variable_values}{A data frame of the  values of the variables that
caused the given SHAP values, generally will be the same data frame or
matrix that was passed to the model for prediction.}

\item{shap_values}{A data frame of shap values, either returned by \code{mshap()}
or obtained from the python \code{{shap}} module.}

\item{expected_value}{The expected value of the SHAP explainer, either
returned by \code{mshap()} or obtained from the python \code{{shap}} module.}

\item{names}{A character vector of variable names, corresponding to the
order of the columns in both \code{variable_values} and \code{shap_values}. If
\code{NULL} (default), then the column names of the \code{variable_values} are
taken as \code{names}.}

\item{num_vars}{An integer specifying the number of variables to show in the
plot, defaults to the 10 most important.}

\item{fill_colors}{A character vector of length 2. The first element
specifies the fill of a negative SHAP value and the second element
specifies the fill of a positive SHAP value.}

\item{connect_color}{A string specifying the color of the line segment that
connects the SHAP value bars}

\item{expected_color}{A string specifying the color of the line that marks
the baseline value, or the expected model output.}

\item{predicted_color}{A string specifying the color of the line that marks
the value predicted by the model.}

\item{title}{A string specifying the title of the plot.}

\item{font_family}{A string specifying the font family, defaults to Times
New Roman.}
}
\value{
A \code{{ggplot2}} object
}
\description{
This Function plots the given contributions for a single observation, and
demonstrates how the model arrived at the prediction for the given
observation.
}
\details{
This function allows the user to pass a sing row from a data frame of SHAP
values and variable values along with an expected model output and it
returns a ggplot object displaying a specific map of the effect of Variable
value on SHAP value. It is created with \code{{ggplot2}}, and the returned value
is a \code{{ggplot2}} object that can be modified for given themes/colors.

Please note that for the \code{variable_values} and \code{shap_values} arguments,
both of which are data frames, the columns must be in the same order.
This is essential in assuring that the variable values and labels are
matched to the correct shap values.
}
\examples{

if (interactive()) {
library(mshap)
library(ggplot2)

# Generate fake data
set.seed(18)
dat <- data.frame(
  age = runif(1000, min = 0, max = 20),
  prop_domestic = runif(1000),
  model = sample(c(0, 1), 1000, replace = TRUE),
  maintain = rexp(1000, .01) + 200
)
shap <- data.frame(
  age = rexp(1000, 1/dat$age) * (-1)^(rbinom(1000, 1, dat$prop_domestic)),
  prop_domestic = -200 * rnorm(100, dat$prop_domestic, 0.02) + 100,
  model = ifelse(dat$model == 0, rnorm(1000, -50, 30), rnorm(1000, 50, 30)),
  maintain = (rnorm(1000, dat$maintain, 100) - 400) * 0.2
)
expected_value <- 1000

# A Basic sumary plot
summary_plot(
  variable_values = dat,
  shap_values = shap
)

# A Customized summary plot
summary_plot(
  variable_values = dat,
  shap_values = shap,
  legend.position = "bottom",
  names = c("Age", "\% Domestic", "Model", "Maintenence Hours"),
  colorscale = c("blue", "purple", "red"),
  font_family = "Arial",
  title = "A Custom Title"
)

# A basic observation plot
observation_plot(
  variable_values = dat[1,],
  shap_values = shap[1,],
  expected_value = expected_value
)

# A Customized Observation plot
observation_plot(
  variable_values = dat[1,],
  shap_values = shap[1,],
  expected_value = expected_value,
  names = c("Age", "\% Domestic", "Model", "Maintenence Hours"),
  font_family = "Arial",
  title = "A Custom Title",
  fill_colors = c("red", "blue"),
  connect_color = "black",
  expected_color = "purple",
  predicted_color = "yellow"
)

# Add elements to the returned object
# see vignette("mshap_plots") for more information
observation_plot(
  variable_values = dat[1,],
  shap_values = shap[1,],
  expected_value = expected_value,
  names = c("Age", "\% Domestic", "Model", "Maintenence Hours"),
  font_family = "Arial",
  title = "A Custom Title"
) +
  geom_label(
    aes(y = 950, x = 4, label = "This is a really big bar!"),
    color = "#FFFFFF",
    fill = NA
  ) +
  theme(
    plot.background = element_rect(fill = "grey"),
    panel.background = element_rect(fill = "lightyellow")
  )
}
}
