% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multilevel.icc.R
\name{multilevel.icc}
\alias{multilevel.icc}
\title{Intraclass Correlation Coefficient, ICC(1) and ICC(2)}
\usage{
multilevel.icc(..., data = NULL, cluster, type = c("1a", "1b", "2"),
               method = c("aov", "lme4", "nlme"), REML = TRUE,
               as.na = NULL, check = TRUE)
}
\arguments{
\item{...}{a numeric vector, matrix, or data frame. Alternatively, an
expression indicating the variable names in \code{data}. Note
that the operators \code{.}, \code{+}, \code{-}, \code{~},
\code{:}, \code{::}, and \code{!} can also be used to select
variables, see 'Details' in the \code{\link{df.subset}}
function.}

\item{data}{a data frame when specifying one or more variables in the
argument \code{...}. Note that the argument is \code{NULL}
when specifying a numeric vector, matrix, or data frame for
the argument \code{...}.}

\item{cluster}{a character string indicating the name of the cluster
variable in \code{...} or \code{data} for two-level data,
a character vector indicating the names of the cluster
variables in \code{...} for three-level data, or a vector
or data frame representing the nested grouping structure
(i.e., group or cluster variables). Alternatively, a
character string or character vector indicating the variable
name(s) of the cluster variable(s) in \code{data}. Note that
the cluster variable at Level 3 come first in a three-level
model, i.e., \code{cluster = c("level3", "level2")}.}

\item{type}{a character string indicating the type of intraclass correlation
coefficient, i.e., \code{type = "1a"} (default) for ICC(1) and
\code{type = "2"} for ICC(2) when specifying a two-level model
(i.e., one cluster variable), and \code{type = "1a"} (default)
for ICC(1) representing the propotion of variance at Level 2
and Level 3, \code{type = "1b"} representing an estimate
of the expected correlation between two randomly chosen elements
in the same group, and \code{type = "2"} for ICC(2) when
specifying a three-level model (i.e., two cluster variables). See 'Details'
for the formula used in this function.}

\item{method}{a character string indicating the method used to estimate
intraclass correlation coefficients, i.e., \code{method = "aov"}
ICC estimated using the \code{aov} function, \code{method = "lme4"}
(default) ICC estimated using the \code{lmer} function in the
\pkg{lme4} package, \code{method = "nlme"} ICC estimated using
the \code{lme} function in the \pkg{nlme} package. Note that
if the lme4 or nlme package is needed when estimating ICCs in
a three-level model.}

\item{REML}{logical: if \code{TRUE} (default), restricted maximum likelihood
is used to estimate the null model when using the \code{lmer}
function in the \pkg{lme4} package or the \code{lme} function
in the \pkg{nlme} package.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting
the analysis. Note that \code{as.na()} function is only applied
to \code{x} but not to \code{cluster}.}

\item{check}{logical: if \code{TRUE} (default), argument specification is
checked.}
}
\value{
Returns a numeric vector or matrix with intraclass correlation coefficient(s).
In a three level model, the label \code{L2} is used for ICCs at Level 2 and \code{L3} for ICCs at Level 3.
}
\description{
This function computes the intraclass correlation coefficient ICC(1), i.e.,
proportion of the total variance explained by the grouping structure, and ICC(2),
i.e., reliability of aggregated variables in a two-level and three-level model.
}
\details{
\describe{
\item{\strong{Two-Level Model}}{In a two-level model, the intraclass
correlation coefficients are computed in the random intercept-only model:

\deqn{Y_{ij} = \gamma_{00} + u_{0j} + r_{ij}}

where the variance in \eqn{Y} is decomposed into two independent components:
\eqn{\sigma^2_{u_{0}}}, which represents the variance at Level 2, and
\eqn{\sigma^2_{r}}, which represents the variance at Level 1 (Hox et al.,
2018). These two variances sum up to the total variance and are referred to
as variance components. The intraclass correlation coefficient, ICC(1)
\eqn{\rho} requested by \code{type = "1a"} represents the proportion of the
total variance explained by the grouping structure and is defined by the equation

\deqn{\rho = \frac{\sigma^2_{u_{0}}}{\sigma^2_{u_{0}} + \sigma^2_{r}}}

The intraclass correlation coefficient, ICC(2) \eqn{\lambda_j} requested by
\code{type = "2"} represents the reliability of aggregated variables and is
defined by the equation

\deqn{\lambda_j = \frac{\sigma^2_{u_{0}}}{\sigma^2_{u_{0}} + \frac{\sigma^2_{r}}{n_j}} = \frac{n_j\rho}{1 + (n_j - 1)\rho}}

where \eqn{n_j} is the average group size (Snijders & Bosker, 2012).
}
\item{\strong{Three-Level Model}}{In a three-level model, the intraclass
correlation coefficients are computed in the random intercept-only model:

\deqn{Y_{ijk} = \gamma_{000} + v_{0k} + u_{0jk} + r_{ijk}}

where the variance in \eqn{Y} is decomposed into three independent components:
\eqn{\sigma^2_{v_{0}}}, which represents the variance at Level 3,
\eqn{\sigma^2_{u_{0}}}, which represents the variance at Level 2, and
\eqn{\sigma^2_{r}}, which represents the variance at Level 1 (Hox et al.,
2018). There are two ways to compute the intraclass correlation coefficient
in a three-level model. The first method requested by \code{type = "1a"}
represents the proportion of variance at Level 2 and Level 3 and should be
used if we are interestd in a decomposition of the variance across levels.
The intraclass correlation coefficient, ICC(1) \eqn{\rho_{L2}} at Level 2 is
defined as:

\deqn{\rho_{L2} = \frac{\sigma^2_{u_{0}}}{\sigma^2_{v_{0}} + \sigma^2_{u_{0}} + \sigma^2_{r}}}

The ICC(1) \eqn{\rho_{L3}} at Level 3 is defined as:

\deqn{\rho_{L3} = \frac{\sigma^2_{v_{0}}}{\sigma^2_{v_{0}} + \sigma^2_{u_{0}} + \sigma^2_{r}}}

The second method requested by \code{type = "1b"} represents the expected
correlation between two randomly chosen elements in the same group. The
intraclass correlation coefficient, ICC(1) \eqn{\rho_{L2}} at Level 2 is
defined as:

\deqn{\rho_{L2} = \frac{\sigma^2_{v_{0}} + \sigma^2_{u_{0}}}{\sigma^2_{v_{0}} + \sigma^2_{u_{0}} + \sigma^2_{r}}}

The ICC(1) \eqn{\rho_L3} at Level 3 is defined as:

\deqn{\rho_{L3} = \frac{\sigma^2_{v_{0}}}{\sigma^2_{v_{0}} + \sigma^2_{u_{0}} + \sigma^2_{r}}}

Note that both formula are correct, but express different aspects of the data,
which happen to coincide when there are only two levels (Hox et al., 2018).

The intraclass correlation coefficients, ICC(2) requested by \code{type = "2"}
represent the reliability of aggregated variables at Level 2 and Level 3.
The ICC(2) \eqn{\lambda_j} at Level 2 is defined as:

\deqn{\lambda_j = \frac{\sigma^2_{u_{0}}}{\sigma^2_{u_{0}} + \frac{\sigma^2_{r}}{n_j}}}

The ICC(2) \eqn{\lambda_k} at Level 3 is defined as:

\deqn{\lambda_k = \frac{\sigma^2_{v_{0}}}{\frac{{\sigma^2_{v_{0}} + \sigma^2_{u_{0}}}}{n_{j}} + \frac{\sigma^2_{r}}{n_k \cdot n_j}}}

where \eqn{n_j} is the average group size at Level 2 and \eqn{n_j} is the average group size at Level 3 (Hox et al., 2018).}
}
}
\examples{
# Load data set "Demo.twolevel" in the lavaan package
data("Demo.twolevel", package = "lavaan")

#----------------------------------------------------------------------------
# Two-Level Models

#..........
# Cluster variable specification

# Example 1a: Cluster variable 'cluster' in '...'
multilevel.icc(Demo.twolevel[, c("y1", "cluster")], cluster = "cluster")

# Example 1b: Cluster variable 'cluster' not in '...'
multilevel.icc(Demo.twolevel$y1, cluster = Demo.twolevel$cluster)

# Example 1c: Alternative specification using the 'data' argument
multilevel.icc(y1, data = Demo.twolevel, cluster = "cluster")

#..........

# Example 2: ICC(1) for 'y1'
multilevel.icc(Demo.twolevel$y1, cluster = Demo.twolevel$cluster)

# Example 3: ICC(2)
multilevel.icc(Demo.twolevel$y1, cluster = Demo.twolevel$cluster, type = 2)

# Example 4: ICC(1)
# use lme() function in the lme4 package to estimate ICC
multilevel.icc(Demo.twolevel$y1, cluster = Demo.twolevel$cluster, method = "nlme")

# Example 5a: ICC(1) for 'y1', 'y2', and 'y3'
multilevel.icc(Demo.twolevel[, c("y1", "y2", "y3")], cluster = Demo.twolevel$cluster)

# Example 5b: Alternative specification using the 'data' argument
multilevel.icc(y1:y3, data = Demo.twolevel, cluster = "cluster")

#----------------------------------------------------------------------------
# Three-Level Models

# Create arbitrary three-level data
Demo.threelevel <- data.frame(Demo.twolevel, cluster2 = Demo.twolevel$cluster,
                                             cluster3 = rep(1:10, each = 250))

#..........
# Cluster variable specification

# Example 6a: Cluster variables 'cluster' in '...'
multilevel.icc(Demo.threelevel[, c("y1", "cluster3", "cluster2")],
               cluster = c("cluster3", "cluster2"))

# Example 6b: Cluster variables 'cluster' not in '...'
multilevel.icc(Demo.threelevel$y1, cluster = Demo.threelevel[, c("cluster3", "cluster2")])

# Example 6c: Alternative specification using the 'data' argument
multilevel.icc(y1, data = Demo.threelevel, cluster = c("cluster3", "cluster2"))

#..........

# Example 7a: ICC(1), propotion of variance at Level 2 and Level 3
multilevel.icc(y1, data = Demo.threelevel, cluster = c("cluster3", "cluster2"))

# Example 7b: ICC(1), expected correlation between two randomly chosen elements
# in the same group
multilevel.icc(y1, data = Demo.threelevel, cluster = c("cluster3", "cluster2"),
               type = "1b")

# Example 7c: ICC(2)
multilevel.icc(y1, data = Demo.threelevel, cluster = c("cluster3", "cluster2"),
type = "2")
}
\references{
Hox, J., Moerbeek, M., & van de Schoot, R. (2018). \emph{Multilevel analysis:
Techniques and applications} (3rd. ed.). Routledge.

Snijders, T. A. B., & Bosker, R. J. (2012). \emph{Multilevel analysis: An introduction
to basic and advanced multilevel modeling} (2nd ed.). Sage Publishers.
}
\seealso{
\code{\link{multilevel.cfa}}, \code{\link{multilevel.cor}},
\code{\link{multilevel.descript}}

}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
