% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fat.r
\name{fat}
\alias{fat}
\title{Regression tests for detecting funnel plot asymmetry}
\usage{
fat(b, b.se, n.total, d.total, d1, d2, method = "E-FIV")
}
\arguments{
\item{b}{Vector with the effect size of each study. Examples are log odds ratio, log hazards ratio, 
log relative risk.}

\item{b.se}{Optional vector with the standard error of the effect size of each study}

\item{n.total}{Optional vector with the total sample size of each study}

\item{d.total}{Optional vector with the total number of observed events for each study}

\item{d1}{Optional vector with the total number of observed events in the exposed groups}

\item{d2}{Optional vector with the total number of observed events in the unexposed groups}

\item{method}{Method for testing funnel plot asymmetry, defaults to \code{"E-FIV"} (Egger's test with 
multiplicative dispersion). Other options are \code{E-UW}, \code{M-FIV}, \code{M-FPV}, \code{D-FIV} and 
\code{D-FAV}. More info in "Details"}
}
\value{
a list containing the following entries:
\describe{
 \item{"pval"}{A two-sided P-value indicating statistical significance of the funnel plot asymettry test. 
 Values below the significance level (usually defined as 10\%) support the presence of funnel plot asymmetry,
 and thus small-study effects.  }
 \item{"model"}{A fitted \code{glm} object, representing the estimated regression model used for testing funnel
 plot asymmetry.}
}
}
\description{
The presence of small-study effects is a common threat to systematic reviews and meta-analyses, especially when 
it is due to publication bias, which occurs when small primary studies are more likely to be reported (published) 
if their findings were positive. The presence of small-study effects can be verified by visual inspection of 
the funnel plot, where for each included study of the meta-analysis, the estimate of the reported effect size is 
depicted against a measure of precision or sample size. 
The premise is that the scatter of plots should reflect a funnel shape, if small-study 
effects do not exist. However, when small studies are predominately in one direction (usually the 
direction of larger effect sizes), asymmetry will ensue.\cr \cr
The \code{\link{fat}} function implements several tests for detecting funnel plot asymmetry, 
which can be used when the presence of between-study heterogeneity in treatment effect is relatively low.
}
\details{
\subsection{Egger regression method}{A common method to test the presence of small-study effects is given as the 
following unweighted regression model (\code{method="E-UW"}, Egger 1997): 
\deqn{\hat{b}_k = \beta_0 + \beta_1\, \widehat \mathrm{SE}(\hat{b}_k) + \epsilon_k \;,\; \epsilon_k \sim \mathcal{N}\left(0, \sigma^2 \right) }{b = B0 + B1*b.se + e;  e~N(0, s^2)}
Whereas \eqn{\beta_0}{B0} indicates the size and direction of the treatment effect, \eqn{\beta_1}{B1} provides 
a measure of asymmetry; the larger its deviation from zero the more pronounced the asymmetry. Otherwise, if 
\eqn{\beta_1=0}{B1=0}, there is no association between the estimated effect sizes \eqn{\hat{b}_k}{b} and their 
corresponding estimates for the standard error \eqn{\widehat \mathrm{SE}(\hat{b}_k)}{b.se} among the reported 
studies, indicating no asymmetry and thus no small-study effects. \cr \cr
It is possible to allow for between-study heterogeneity by adopting a multiplicative overdispersion parameter 
by which the variance in each study is multiplied (\code{method="E-FIV"}, Sterne 2000):
\deqn{\hat{\beta}_k = a + b\, \widehat \mathrm{SE}(\hat{\beta}_k) + \epsilon_k \;,\; \epsilon_k \sim \mathcal{N}(0, \phi \; \widehat \mathrm{var}(\hat{\beta}_k))}{b = B0 + B1*b.se + e;  e~N(0, P*b.se^2)}
Unfortunately, both tests are known to be intrinsically biased because: (i) the independent variable is subject to sampling 
variability; (ii) the standardized treatment effect is correlated with its estimated precision; and 
(iii) for binary data, the independent regression variable is a biased estimate of the true precision, 
with larger bias for smaller sample sizes (Macaskill 2001).
}

\subsection{Macaskill regression method}{
To overcome the problems with the Egger approach, Macaskill et al. consider fitting a regression directly
to the data using the treatment effect as the dependent variable, and study size (\eqn{n_k}{n.total}) as the 
independent variable. Again, the observations are weighted by the inverse variance of the estimate
to allow for possible heteroscedasticity  (\code{method="M-FIV"}, Macaskill 2001):
\deqn{\hat{\beta}_k = a + b \,n_k + \epsilon_k \;,\; \epsilon_k \sim \mathcal{N}(0, \phi \; \widehat \mathrm{var}(\hat{\beta}_k))}{b = B0 + B1*n.total + e;  e~N(0, P*b.se^2)}
Macaskill et al. also proposed an alternative test where a 'pooled' estimate of the outcome proportion is used
for the variance \eqn{\widehat \mathrm{var}(\hat{b}_k)}{b.se^2} (\code{method="M-FPV"}, Macaskill 2001):
\deqn{\hat{\beta}_k = a + b \,n_k + \epsilon_k \;,\; \epsilon_k \sim \mathcal{N}\left(0, \phi \; \frac{1}{d_k (1-d_k/n_k)}\right)}{b = B0 + B1*n.total + e;  e~N(0, P/(d.total * (1-d.total/n.total)))}
For studies with zero events, a continuity correction is applied by adding 0.5 to all cell counts.
}

\subsection{Peters regression method}{
A modification of Macaskill's test was proposed by Peters et al. to obtain more balanced type-I error rates 
in the tail probability areas  (\code{method="P-FPV"}, Peters 2006):
\deqn{\hat{\beta}_k = a + b \,\frac{1}{n_k} + \epsilon_k \;,\; \epsilon_k \sim \mathcal{N}\left(0, \phi \; \frac{1}{d_k (1-d_k/n_k)}\right)}{b = B0 + B1/n.total + e;  e~N(0, P/(d.total * (1-d.total/n.total)))}
Again, 0.5 is added to all cells for studies with zero events.
}

\subsection{Debray regression method}{
Because the use of aforementioned tests may be less appropriate in the presence of survival data, Debray et al. 
proposed using the total number of events (\eqn{d_k}{d.total}) as independent variable (\code{method="D-FIV"}, Debray 2017):
\deqn{\hat{\beta}_k = a + b\, \frac{1}{d_k} + \epsilon_k  \;,\; \epsilon_k \sim \mathcal{N}(0, \phi \; \widehat \mathrm{var}(\hat{\beta}_k))}{b = B0 + B1/d.total + e;  e~N(0, P*b.se^2)}
For studies with zero events, the total number of observed events is set to 1.
Alternatively, when \eqn{\widehat \mathrm{var}(\hat{\beta}_k)}{b.se} is unknown or derived from small samples, 
Debray at al.proposed to use the following regression model (\code{method="D-FAV"}, Debray 2017):
\deqn{\hat{\beta}_k = a + b\, \frac{1}{d_k} + \epsilon_k  \;,\; \epsilon_k \sim \mathcal{N}\left(0, \phi \; \left(\frac{1}{d_{k1}}+\frac{1}{d_{k2}}\right)\right)}{b = B0 + B1/d.total + e;  e~N(0, P/(1/d1 + 1/d2))}
}
}
\examples{
data(Fibrinogen)
b <- log(Fibrinogen$HR)
b.se <- ((log(Fibrinogen$HR.975) - log(Fibrinogen$HR.025))/(2*qnorm(0.975)))
n.total <- Fibrinogen$N.total
d.total <- Fibrinogen$N.events

fat(b=b, b.se=b.se)
fat(b=b, b.se=b.se, d.total=d.total, method="D-FIV")

# Note that many tests are also available via metafor
require(metafor)
fat(b=b, b.se=b.se, n.total=n.total, method="M-FIV")
regtest(x=b, sei=b.se, ni=n.total, model="lm", predictor="ni") 

}
\references{
Debray TPA, Moons KGM, Riley RD. Detecting small-study effects and funnel plot asymmetry in meta-analysis of 
survival data: a comparison of new and existing tests. Res Syn Meth. 2018;9(1):41--50.\cr
\cr
Egger M, Davey Smith G, Schneider M, Minder C. Bias in meta-analysis detected by a simple, graphical test. 
\emph{BMJ}. 1997;315(7109):629--34. \cr
\cr
Macaskill P, Walter SD, Irwig L. A comparison of methods to detect publication bias in meta-analysis. 
\emph{Stat Med}. 2001;20(4):641--54.\cr 
\cr
Peters JL, Sutton AJ, Jones DR, Abrams KR, Rushton L. Comparison of two methods to detect publication bias 
in meta-analysis. \emph{JAMA}. 2006 Feb 8;295(6):676--80.\cr
\cr 
Sterne JA, Gavaghan D, Egger M. Publication and related bias in meta-analysis: power of statistical tests 
and prevalence in the literature. \emph{J Clin Epidemiol}. 2000;53(11):1119--29.
}
\seealso{
\code{\link{plot.fat}}
}
\author{
Thomas Debray <thomas.debray@gmail.com>
}
