% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parse.r
\name{parse_date_time}
\alias{fast_strptime}
\alias{parse_date_time}
\alias{parse_date_time2}
\title{Parse character and numeric date-time vectors with user friendly order
formats.}
\usage{
parse_date_time(x, orders, tz = "UTC", truncated = 0, quiet = FALSE,
  locale = Sys.getlocale("LC_TIME"), select_formats = .select_formats,
  exact = FALSE)

parse_date_time2(x, orders, tz = "UTC", exact = FALSE, lt = FALSE)

fast_strptime(x, format, tz = "UTC", lt = TRUE)
}
\arguments{
\item{x}{a character or numeric vector of dates}

\item{orders}{a character vector of date-time formats. Each order string is
series of formatting characters as listed \code{\link[base]{strptime}} but
might not include the "\%" prefix, for example "ymd" will match all the
possible dates in year, month, day order.  Formatting orders might include
arbitrary separators. These are discarded.  See details for implemented
formats.}

\item{tz}{a character string that specifies the time zone with which to
parse the dates}

\item{truncated}{integer, number of formats that can be missing. The most
common type of irregularity in date-time data is the truncation due to
rounding or unavailability of the time stamp. If \code{truncated}
parameter is non-zero \code{parse_date_time} also checks for truncated
formats. For example,  if the format order is "ymdhms" and \code{truncated
= 3}, \code{parse_date_time} will correctly parse incomplete dates like
\code{2012-06-01 12:23}, \code{2012-06-01 12} and
\code{2012-06-01}. \bold{NOTE:} \code{ymd} family of functions are based
on \code{strptime} which currently fails to parse \code{\%y-\%m} formats.}

\item{quiet}{logical. When TRUE progress messages are not printed, and
"no formats found" error is surpresed and the function simply returns a
vector of NAs.  This mirrors the behavior of base R functions
\code{strptime} and \code{as.POSIXct}. Default is FALSE.}

\item{locale}{locale to be used, see \link{locales}. On linux systems you
can use \code{system("locale -a")} to list all the installed locales.}

\item{select_formats}{A function to select actual formats for parsing from a
set of formats which matched a training subset of \code{x}. it receives a
named integer vector and returns a character vector of selected
formats. Names of the input vector are formats (not orders) that matched
the training set. Numeric values are the number of dates (in the training
set) that matched the corresponding format. You should use this argument
if the default selection method fails to select the formats in the right
order. By default the formats with most formating tockens (\%) are
selected and \%Y counts as 2.5 tockens (so that it has a priority over
\%y\%m). Se examples.}

\item{exact}{logical. If \code{TRUE}, \code{orders} parameter is interpreted
as an exact \code{strptime} format and no trainign or guessing are
performed.}

\item{lt}{logical. If TRUE returned object is of class POSIXlt, and POSIXct
otherwise. For compatibility with base `strptime` function default is TRUE
for `fast_strptime` and FALSE for `parse_date_time2`.}

\item{format}{a character string of formats. It should include all the
separators and each format must be prefixed with %, just as in the format
argument of \code{strptime}.}
}
\value{
a vector of POSIXct date-time objects
}
\description{
\code{parse_date_time} parses an input vector into POSIXct date-time
object. It differs from \code{\link[base]{strptime}} in two respects. First,
it allows specification of the order in which the formats occur without the
need to include separators and "\%" prefix. Such a formating argument is
refered to as "order". Second, it allows the user to specify several
format-orders to handle heterogeneous date-time character
representations. \cr \code{parse_date_time2} is a fast C parser of numeric
orders. \cr \code{fast_strptime} is a fast C parser of numeric formats only
that accepts explicit format arguments, just as
\code{\link[base]{strptime}}.
}
\details{
When several format-orders are specified \code{parse_date_time} sorts the
supplied format-orders based on a training set and then applies them
recursively on the input vector.

\code{parse_date_time}, and all derived functions, such as \code{ymd_hms},
\code{ymd} etc, will drop into \code{fast_strptime} instead of
\code{strptime} whenever the guesed from the input data formats are all
numeric.

The list below contains formats recognized by lubridate. For numeric formats
leading 0s are optional. In contrast to \code{strptime}, some of the formats
have been extended for efficiency reasons. They are marked with "*". Fast
perasers,  \code{parse_date_time2} and \code{fast_strptime}, currently
accept only formats marked with "!".

\describe{ \item{\code{a}}{Abbreviated weekday name in the current
locale. (Also matches full name)}

\item{\code{A}}{Full weekday name in the current locale.  (Also matches
abbreviated name).

You need not specify \code{a} and \code{A} formats explicitly. Wday is
automatically handled if \code{preproc_wday = TRUE}}

\item{\code{b}}{Abbreviated month name in the current locale.  (Also matches full name.)}

\item{\code{B}}{Full month name in the current locale.  (Also matches abbreviated name.)}

\item{\code{d}!}{Day of the month as decimal number (01--31 or 0--31)}

\item{\code{H}!}{Hours as decimal number (00--24 or 0--24).}

\item{\code{I}}{Hours as decimal number (01--12 or 1--12).}

\item{\code{j}}{Day of year as decimal number (001--366 or 1--366).}

\item{\code{m}!*}{Month as decimal number (01--12 or 1--12). For
\code{parse_date_time}, also matches abbreviated and full months names as
\code{b} and \code{B} formats.}

\item{\code{M}!}{Minute as decimal number (00--59 or 0--59).}

\item{\code{p}}{AM/PM indicator in the locale.  Used in
                  conjunction with \code{I} and \bold{not} with \code{H}.  An
                  empty string in some locales.}

\item{\code{S}!}{Second as decimal number (00--61 or 0--61), allowing for up
to two leap-seconds (but POSIX-compliant implementations will ignore leap
seconds).}

\item{\code{OS}}{Fractional second.}

\item{\code{U}}{Week of the year as decimal number (00--53 or 0-53) using
Sunday as the first day 1 of the week (and typically with the first Sunday
of the year as day 1 of week 1).  The US convention.}

\item{\code{w}}{Weekday as decimal number (0--6, Sunday is 0).}

\item{\code{W}}{Week of the year as decimal number (00--53 or 0-53) using
Monday as the first day of week (and typically with the first Monday of the
year as day 1 of week 1).  The UK convention.}

\item{\code{y}!*}{Year without century (00--99 or 0--99).  In
\code{parse_date_time} also matches year with century (Y format).}

\item{\code{Y}!}{Year with century.}

\item{\code{z}!*}{ISO8601 signed offset in hours and minutes from UTC. For
example \code{-0800}, \code{-08:00} or \code{-08}, all represent 8 hours
behind UTC. This format also matches the Z (Zulu) UTC indicator. Because
strptime doesn't fully support ISO8601 this format is implemented as an
union of 4 orders: Ou (Z), Oz (-0800), OO (-08:00) and Oo (-08). You can use
these four orders as any other but it is rarely
necessary. \code{parse_date_time2} and \code{fast_strptime} support all of
the timezone formats.}

\item{\code{r}*}{Matches \code{Ip} and \code{H} orders.}
\item{\code{R}*}{Matches \code{HM} and\code{IMp} orders.}
\item{\code{T}*}{Matches \code{IMSp}, \code{HMS}, and \code{HMOS} orders.}
}
}
\note{
\code{parse_date_time} (and the derivatives \code{ymb}, \code{ymd_hms}
  etc) rely on a sparse guesser that takes at most 501 elements from the
  supplied character vector in order to identify appropriate formats from
  the supplied orders. If you get the error \code{All formats failed to
  parse} and you are confident that your vector contains valid dates, you
  should either set \code{exact} argument to TRUE or use functions that
  don't perform format guessing (\code{fast_strptime},
  \code{parse_date_time2} or \code{strptime} ).

For performance reasons, when timezone is not UTC,
  \code{parse_date_time2} and \code{fast_strptime} perform no validity
  checks for daylight savings time. Thus, if your input string contains an
  invalid date time which falls into DST gap and \code{lt=TRUE} you will get
  an \code{POSIXlt} object with a non-existen time. If \code{lt=FALSE} your
  time instant will be adjusted to a valid time by adding an hour. See
  examples. If you want to get NA for invalid date-times use
  \code{\link{fit_to_timeline}} explicitely.
}
\examples{

## ** orders are much easier to write **
x <- c("09-01-01", "09-01-02", "09-01-03")
parse_date_time(x, "ymd")
parse_date_time(x, "y m d")
parse_date_time(x, "\%y\%m\%d")
#  "2009-01-01 UTC" "2009-01-02 UTC" "2009-01-03 UTC"

## ** heterogenuous date-times **
x <- c("09-01-01", "090102", "09-01 03", "09-01-03 12:02")
parse_date_time(x, c("ymd", "ymd HM"))

## ** different ymd orders **
x <- c("2009-01-01", "02022010", "02-02-2010")
parse_date_time(x, c("dmY", "ymd"))
##  "2009-01-01 UTC" "2010-02-02 UTC" "2010-02-02 UTC"

## ** truncated time-dates **
x <- c("2011-12-31 12:59:59", "2010-01-01 12:11", "2010-01-01 12", "2010-01-01")
parse_date_time(x, "Ymd HMS", truncated = 3)
parse_date_time(x, "ymd_hms", truncated = 3)
## [1] "2011-12-31 12:59:59 UTC" "2010-01-01 12:11:00 UTC"
## [3] "2010-01-01 12:00:00 UTC" "2010-01-01 00:00:00 UTC"

## ** specifying exact formats and avoiding training and guessing **
parse_date_time(x, c("\%m-\%d-\%y", "\%m\%d\%y", "\%m-\%d-\%y \%H:\%M"), exact = TRUE)
## [1] "2001-09-01 00:00:00 UTC" "2002-09-01 00:00:00 UTC" NA "2003-09-01 12:02:00 UT
parse_date_time(c('12/17/1996 04:00:00','4/18/1950 0130'),
                c('\%m/\%d/\%Y \%I:\%M:\%S','\%m/\%d/\%Y \%H\%M'), exact = TRUE)
## [1] "1996-12-17 04:00:00 UTC" "1950-04-18 01:30:00 UTC"

## ** fast parsing **
\dontrun{
  options(digits.secs = 3)
  ## random times between 1400 and 3000
  tt <- as.character(.POSIXct(runif(1000, -17987443200, 32503680000)))
  tt <- rep.int(tt, 1000)

  system.time(out <- as.POSIXct(tt, tz = "UTC"))
  system.time(out1 <- ymd_hms(tt)) # constant overhead on long vectors
  system.time(out2 <- parse_date_time2(tt, "YmdHMOS"))
  system.time(out3 <- fast_strptime(tt, "\%Y-\%m-\%d \%H:\%M:\%OS"))

  all.equal(out, out1)
  all.equal(out, out2)
  all.equal(out, out3)
}

## ** how to use `select_formats` argument **
## By default \%Y has precedence:
parse_date_time(c("27-09-13", "27-09-2013"), "dmy")
## [1] "13-09-27 UTC"   "2013-09-27 UTC"

## to give priority to \%y format, define your own select_format function:

my_select <-   function(trained){
   n_fmts <- nchar(gsub("[^\%]", "", names(trained))) + grepl("\%y", names(trained))*1.5
   names(trained[ which.max(n_fmts) ])
}

parse_date_time(c("27-09-13", "27-09-2013"), "dmy", select_formats = my_select)
## [1] "2013-09-27 UTC" "2013-09-27 UTC"

## ** invalid times with "fast" parcing **
parse_date_time("2010-03-14 02:05:06",  "YmdHMS", tz = "America/New_York")
## [1] NA
parse_date_time2("2010-03-14 02:05:06",  "YmdHMS", tz = "America/New_York")
## [1] "2010-03-14 03:05:06 EDT"
parse_date_time2("2010-03-14 02:05:06",  "YmdHMS", tz = "America/New_York", lt = TRUE)
## [1] "2010-03-14 02:05:06 America/New_York"
}
\seealso{
\code{strptime}, \code{\link{ymd}}, \code{\link{ymd_hms}}
}
\keyword{chron}

