% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multild.R
\name{mldest}
\alias{mldest}
\title{Estimate all pair-wise LD's in a collection of SNPs using genotypes or
genotype likelihoods.}
\usage{
mldest(
  geno,
  K,
  nc = 1,
  type = c("hap", "comp"),
  model = c("norm", "flex"),
  pen = ifelse(type == "hap", 2, 1),
  se = TRUE
)
}
\arguments{
\item{geno}{One of two possible inputs:
\itemize{
\item{A matrix of genotypes (allele dosages). The rows index the
SNPs and the columns index the individuals. That is,
\code{genomat[i, j]} is the allele dosage for individual
\code{j} in SNP \code{i}. When \code{type = "comp"}, the
dosages are allowed to be continuous (e.g. posterior
mean genotypes).}
\item{A three-way array of genotype \emph{log}-likelihoods.
The first dimension indexes the SNPs, the second dimension
indexes the individuals, and the third dimension indexes
the genotypes. That is, \code{genolike_array[i, j, k]}
is the genotype log-likelihood at SNP \code{i} for
individual \code{j} and dosage \code{k - 1}.}
}}

\item{K}{The ploidy of the species. Assumed to be the same for all
individuals.}

\item{nc}{The number of computing cores to use. This should never be
more than the number of cores available in your computing environment.
You can determine the maximum number of available cores by running
\code{parallel::detectCores()} in R. This is probably fine for a
personal computer, but some environments are only
able to use fewer. Ask your admins if you are unsure.}

\item{type}{The type of LD to calculate. The available types are
haplotypic LD (\code{type = "hap"}) or composite LD
(\code{type = "comp"}). Haplotypic LD is only appropriate for
autopolyploids when the individuals are in Hardy-Weinberg
equilibrium (HWE). The composite
measures of LD are always applicable, and consistently estimate the
usual measures of LD when HWE is fulfilled in autopolyploids.
However, when HWE is not fulfilled, interpreting the
composite measures of LD could be a little tricky.}

\item{model}{When \code{type = "comp"} and using genotype likelihoods,
should we use the proportional
bivariate normal model to estimate the genotype distribution
(\code{model = "norm"}), or the general categorical distribution
(\code{model = "flex"})? Defaults to \code{"norm"}.}

\item{pen}{The penalty to be applied to the likelihood. You can think about
this as the prior sample size. Should be greater than 1. Does not
apply if \code{model = "norm"}, \code{type = "comp"}, and using
genotype likelihoods. Also does not apply when \code{type = "comp"}
and using genotypes.}

\item{se}{A logical. Should we calculate standard errors (\code{TRUE}) or
not (\code{FALSE}). Calculating standard errors can be really slow
when \code{type = "comp"}, \code{model = "flex"}, and when using
genotype likelihoods. Otherwise, standard error calculations
should be pretty fast.}
}
\value{
A data frame of class \code{c("lddf", "data.frame")}
with some or all of the following elements:
\describe{
\item{\code{i}}{The index of the first SNP.}
\item{\code{j}}{The index of the second SNP.}
\item{\code{snpi}}{The row name corresponding to SNP \code{i}, if
row names are provided.}
\item{\code{snpj}}{The row name corresponding to SNP \code{j}, if
row names are provided.}
\item{\code{D}}{The estimate of the LD coefficient.}
\item{\code{D_se}}{The standard error of the estimate of
the LD coefficient.}
\item{\code{r2}}{The estimate of the squared Pearson correlation.}
\item{\code{r2_se}}{The standard error of the estimate of the
squared Pearson correlation.}
\item{\code{r}}{The estimate of the Pearson correlation.}
\item{\code{r_se}}{The standard error of the estimate of the
Pearson correlation.}
\item{\code{Dprime}}{The estimate of the standardized LD
coefficient. When \code{type} = "comp", this corresponds
to the standardization where we fix allele frequencies.}
\item{\code{Dprime_se}}{The standard error of \code{Dprime}.}
\item{\code{Dprimeg}}{The estimate of the standardized LD
coefficient. This corresponds to the standardization where
we fix genotype frequencies.}
\item{\code{Dprimeg_se}}{The standard error of \code{Dprimeg}.}
\item{\code{z}}{The Fisher-z transformation of \code{r}.}
\item{\code{z_se}}{The standard error of the Fisher-z
transformation of \code{r}.}
\item{\code{p_ab}}{The estimated haplotype frequency of ab.
Only returned if estimating the haplotypic LD.}
\item{\code{p_Ab}}{The estimated haplotype frequency of Ab.
Only returned if estimating the haplotypic LD.}
\item{\code{p_aB}}{The estimated haplotype frequency of aB.
Only returned if estimating the haplotypic LD.}
\item{\code{p_AB}}{The estimated haplotype frequency of AB.
Only returned if estimating the haplotypic LD.}
\item{\code{q_ij}}{The estimated frequency of genotype i at locus 1
and genotype j at locus 2. Only returned if estimating the
composite LD.}
\item{\code{n}}{The number of individuals used to estimate pairwise LD.}
}
}
\description{
This function is a wrapper to run \code{\link{ldest}()} for many pairs of
SNPs. This takes a maximum likelihood approach to LD estimation. See
\code{\link{ldfast}()} for a method-of-moments approach to LD estimation.
Support is provided for parallelization through the foreach and doParallel
packages. See Gerard (2021) for details.
}
\details{
See \code{\link{ldest}()} for details on the different types of LD
estimators supported.
}
\examples{
set.seed(1)

## Simulate genotypes when true correlation is 0
nloci <- 5
nind  <- 100
K <- 6
nc <- 1
genomat <- matrix(sample(0:K, nind * nloci, TRUE), nrow = nloci)

## Composite LD estimates
lddf <- mldest(geno = genomat,
               K = K,
               nc = nc,
               type = "comp")
lddf[1:6, 1:6]


}
\references{
\itemize{
\item{Gerard, David. "Pairwise Linkage Disequilibrium Estimation for Polyploids." \emph{Molecular Ecology Resources}. Accepted Author Manuscript. (2021) \doi{10.1111/1755-0998.13349}}
}
}
\seealso{
\describe{
\item{\code{\link{ldfast}()}}{Fast, moment-based approach to LD estimation
that also accounts for genotype uncertainty.}
\item{\code{\link{ldest}()}}{For the base function that estimates
pairwise LD.}
\item{\code{\link{sldest}()}}{For estimating pairwise LD along a
sliding window.}
\item{\code{\link{format_lddf}()}}{For formatting the output of
\code{mldest()} as a matrix.}
\item{\code{\link{plot.lddf}()}}{For plotting the output of
\code{mldest()}.}
}
}
\author{
David Gerard
}
