% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jackstraw.R
\name{jackstraw}
\alias{jackstraw}
\title{Non-Parametric Jackstraw (Wrapper)}
\usage{
jackstraw(dat, method = "PCA", FUN = NULL, r = NULL, ...)
}
\arguments{
\item{dat}{a data matrix with \code{m} rows as variables and \code{n} columns as observations.}

\item{method}{a LV estimation method (by default, \code{"PCA"}). Use an optional argument \code{FUN} to specify a custom method.}

\item{FUN}{optionally, provide a specfic function to estimate LVs. Must output \code{r} estimated LVs in a \code{n*r} matrix.}

\item{r}{a number of significant latent variables.}

\item{...}{optional arguemtns passed along to a specific jackstraw function.}
}
\value{
\code{jackstraw} returns a list consisting of
\item{p.value}{\code{m} p-values of association tests between variables and their principal components}
\item{obs.stat}{\code{m} observed F-test statistics}
\item{null.stat}{\code{s*B} null F-test statistics}
}
\description{
Estimates statistical significance of association between variables and their latent variables (LVs).
}
\details{
This is a wrapper for a few different functions using the jackstraw method.
Overall, it computes \code{m} p-values of association between \code{m} variables and their LVs.
Its resampling strategy accounts for the over-fitting characteristics due to direct computation of LVs from the observed data
and protects against an anti-conservative bias.

For advanced use, one may consider computing association between variables and a subset of \code{r} estimated LVs.
For example, when there may be \code{r=3} significant PCs,
a user can carry out significance tests for the top two PCs (while adjusting for the third PC), by specifying \code{r1=c(1,2)} and \code{r=3}.

Please take a careful look at your data and use appropriate graphical and statistical criteria
to determine a number of interesting/significant LVs, \code{r}.
It is assumed that \code{r} latent variables account for systematic variation in the data.

For advanced usage, see \code{jackstraw.PCA}, \code{jackstraw.LFA}, and \code{jackstraw.FUN}.

If \code{s} is not supplied, \code{s} is set to about 10\% of \code{m} variables.
If \code{B} is not supplied, \code{B} is set to \code{m*10/s}.
}
\section{Optional Arguments (see linked functions)}{

\describe{
  \item{s}{a number of ``synthetic'' null variables. Out of \code{m} variables, \code{s} variables are independently permuted.}
  \item{B}{a number of resampling iterations.}
  \item{r1}{a numeric vector of latent variables (e.g., PCs) of interest. Not appropriate for all methods or functions.}
  \item{covariate}{a model matrix of covariates with \code{n} observations. Must include an intercept in the first column. Not appropriate for all methods and functions.}
  \item{verbose}{a logical specifying to print the computational progress. By default, \code{FALSE}.}
  \item{seed}{a seed for the random number generator.}
}
}

\examples{
set.seed(1234)
## simulate data from a latent variable model: Y = BL + E
B = c(rep(1,50),rep(-1,50), rep(0,900))
L = rnorm(20)
E = matrix(rnorm(1000*20), nrow=1000)
dat = B \%*\% t(L) + E
dat = t(scale(t(dat), center=TRUE, scale=TRUE))

## apply the jackstraw
out = jackstraw(dat, r=1, method="PCA")

## Use optional arguments
## For example, set s and B for a balance between speed of the algorithm and accuracy of p-values
\dontrun{
out = jackstraw(dat, r=1, s=10, B=1000, seed=5678)
}
}
\references{
Chung and Storey (2013) Statistical significance of variables driving systematic variation in high-dimensional data Bioinformatics, 31(4): 545-554 \url{http://bioinformatics.oxfordjournals.org/content/31/4/545}
}
\seealso{
\link{permutationPA} \link{jackstraw.PCA} \link{jackstraw.LFA} \link{jackstraw.FUN}
}
\author{
Neo Christopher Chung \email{nchchung@gmail.com}
}
