% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/intsel.R
\name{intsel}
\alias{intsel}
\title{Logistic regression with two-way interaction screening}
\usage{
intsel(
  x,
  y,
  intercept = TRUE,
  p.screen,
  lambda,
  par_init,
  stepsize_init = 1,
  stepsize_shrink = 0.8,
  tol = 1e-05,
  maxit = 1000L,
  verbose = FALSE
)
}
\arguments{
\item{x}{Predictor matrix with dimension \eqn{n * p}, where \eqn{n} is the number of subjects, and \eqn{p} is the number of predictors.}

\item{y}{Binary outcome, a vector of length \eqn{n}.}

\item{intercept}{Logical, indicating whether an intercept term should be included in the model. The intercept term will not be penalized. The default is \code{TRUE}.}

\item{p.screen}{Number of variables of which all two-way interactions are screened. These variables should be placed in the \code{p.screen} left-most columns of matrix \code{x}.}

\item{lambda}{Sequence of regularization coefficients \eqn{\lambda}'s. Will be sorted in a decreasing order.}

\item{par_init}{Optional, vector of initial values of the optimization algorithm. Default initial value is zero for all \eqn{p} variables.}

\item{stepsize_init}{Initial value of the stepsize of the optimization algorithm. Default is 1.0.}

\item{stepsize_shrink}{Factor in \eqn{(0,1)} by which the stepsize shrinks in the backtracking linesearch. Default is 0.8.}

\item{tol}{Convergence criterion. Algorithm stops when the \eqn{l_2} norm of the parameter update is smaller than \code{tol}. Default is \code{1e-5}.}

\item{maxit}{Maximum number of iterations allowed. Default is \code{100L}.}

\item{verbose}{Logical, whether progress is printed. Default is \code{FALSE}.}
}
\value{
A list.
  \item{lambdas}{The user-specified regularization coefficients \code{lambda} sorted in decreasing order.}
  \item{estimates}{A matrix, with each column corresponding to the coefficient estimates at each \eqn{\lambda} in \code{lambdas}.}
  \item{iterations}{A vector of number of iterations it takes to converge at each \eqn{\lambda} in \code{lambdas}.}
  \item{x.original}{The input matrix \code{x}.}
  \item{x}{The predictor matrix with \code{x} plus \code{p.screen} * (\code{p.screen} - 1)/2 interaction terms.}
  \item{y}{The input \code{y}.}
  \item{p.screen}{The input \code{p.screen}.}
  \item{intercept}{The input \code{intercept}.}
}
\description{
Fit a logistic regression model including all the two-way interaction terms between the user-specified set of variables. The method uses an overlapping group lasso penalty that respects the commonly recognized selection rule, which is that, when the interaction term is selected into the model, both main effect terms should be in the model too. The regularization path is computed at a grid of values for the regularization parameter lambda.
}
\examples{
n <- 1000
p.int <- 5
p.noint <- 3
intercept <- TRUE
p.screen <- 5

p.int.expand <- p.int*(p.int-1)/2
p.main <- p.int + p.noint
x <- matrix(rnorm(n * p.main), nrow = n, ncol = p.main)

# true model
# logit(p) = 0.1 + 0.3 x1 + 0.3 x2 + 0.3 x8 + 0.2 * x1 * x2

beta.true <- rep(0, p.main)
beta.true[c(1, 2, p.main)] <- 0.3
eta <- x \%*\% beta.true + 0.2 * x[, 1] * x[, 2]

if (intercept) eta <- eta + 0.1

py <- 1/(1 + exp(-eta))

y <- rbinom(n, 1, py)

nlam <- 30
lambdas <- exp(seq(log(0.1), log(0.00005), length.out = nlam))

# All the pairwise two-way interactions for the first p.screen variables 
# are included in the model and screened in a data-driven manner.
fit <- intsel(x = x,
              y = y,
              p.screen = 5,
              intercept = intercept,
              lambda = lambdas)
fit$iterations
fit$estimates[, 1]
}
