% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_datagrid.R
\name{get_datagrid}
\alias{get_datagrid}
\alias{get_datagrid.data.frame}
\alias{get_datagrid.numeric}
\alias{get_datagrid.factor}
\alias{get_datagrid.default}
\title{Create a reference grid}
\usage{
get_datagrid(x, ...)

\method{get_datagrid}{data.frame}(
  x,
  at = "all",
  factors = "reference",
  numerics = "mean",
  preserve_range = FALSE,
  reference = x,
  length = 10,
  range = "range",
  ...
)

\method{get_datagrid}{numeric}(x, length = 10, range = "range", ...)

\method{get_datagrid}{factor}(x, ...)

\method{get_datagrid}{default}(
  x,
  at = "all",
  factors = "reference",
  numerics = "mean",
  preserve_range = TRUE,
  reference = x,
  include_smooth = TRUE,
  include_random = FALSE,
  include_response = FALSE,
  data = NULL,
  ...
)
}
\arguments{
\item{x}{An object from which to construct the reference grid.}

\item{...}{Arguments passed to or from other methods (for instance, \code{length}
or \code{range} to control the spread of numeric variables.).}

\item{at}{Can be \code{"all"}, a character vector or list of named elements, indicating
the predictors of interest (focal predictors). Can also contain assignments
(as named list, e.g. \code{at = list(c(Sepal.Length = c(2, 4), Species = "setosa"))},
or as string, e.g. \code{at = "Sepal.Length = 2"} or
\code{at = c("Sepal.Length = 2", "Species = 'setosa'")} - note the usage of single
and double quotes to assign strings within strings). The remaining variables
will be fixed.}

\item{factors}{Type of summary for factors. Can be \code{"reference"} (set at the
reference level), \code{"mode"} (set at the most common level) or \code{"all"} to
keep all levels.}

\item{numerics}{Type of summary for numeric values. Can be \code{"all"} (will
duplicate the grid for all unique values), any function (\code{"mean"},
\code{"median"}, ...) or a value (e.g., \code{numerics = 0}).}

\item{preserve_range}{In the case of combinations between numeric variables
and factors, setting \code{preserve_range = TRUE} will drop the observations
where the value of the numeric variable is originally not present in the
range of its factor level. This leads to an unbalanced grid. Also, if you
want the minimum and the maximum to closely match the actual ranges, you
should increase the \code{length} argument.}

\item{reference}{The reference vector from which to compute the mean and SD.
Used when standardizing or unstandardizing the grid using \code{effectsize::standardize}.}

\item{length}{Length of numeric target variables selected in \code{"at"}. This arguments
controls the number of (equally spread) values that will be taken to represent the
continuous variables. A longer length will increase precision, but can also
substantially increase the size of the datagrid (especially in case of interactions).
If \code{NA}, will return all the unique values. In case of multiple continuous target
variables, \code{length} can also be a vector of different values (see examples).}

\item{range}{If \code{"range"} (default), will use the minimum and maximum of the
original data vector as end-points (min and max). If an interval type is specified,
such as \code{\link[=IQR]{"iqr"}}, \code{\link[bayestestR:ci]{"ci"}}, \code{\link[bayestestR:hdi]{"hdi"}} or \code{\link[bayestestR:eti]{"eti"}}, it will spread the values within
that range (the default CI width is \verb{95\%} but this can be changed by adding for instance
\code{ci = 0.90}. See \code{\link[=IQR]{IQR()}} and \code{\link[bayestestR:ci]{bayestestR::ci()}}. This can be useful to have
more robust change and skipping extreme values. If \code{\link[=sd]{"sd"}} or \code{\link[=mad]{"mad"}}, it will
spread by this dispersion index around the mean or the median, respectively.
If the \code{length} argument is an even number (e.g., \code{4}), it will have one more
step on the positive side (i.e., \verb{-1, 0, +1, +2}). The result is a named vector.
See examples.}

\item{include_smooth}{If \code{x} is a model object, decide whether smooth terms
should be included in the data grid or not.}

\item{include_random}{If \code{x} is a mixed model object, decide whether random
effect terms should be included in the data grid or not. If
\code{include_random} is \code{FALSE}, but \code{x} is a mixed model with random effects,
these will still be included in the returned grid, but set to their
"population level" value (e.g., \code{NA} for \emph{glmmTMB} or \code{0} for \emph{merMod}).
This ensures that common \code{predict()} methods work properly, as these
usually need data with all variables in the model included.}

\item{include_response}{If \code{x} is a model object, decide whether the response
variable should be included in the data grid or not.}

\item{data}{Optional, the data frame that was used to fit the model. Usually,
the data is retrieved via \code{get_data()}.}
}
\value{
Reference grid data frame.
}
\description{
Create a reference matrix, useful for visualisation, with evenly spread and
combined values. Usually used to make generate predictions using \code{\link[=get_predicted]{get_predicted()}}.
See this \href{https://easystats.github.io/modelbased/articles/visualisation_matrix.html}{vignette} for a tutorial on how to create a visualisation matrix using this function.
}
\examples{
# Datagrids of variables and dataframes =====================================
if (require("bayestestR", quietly = TRUE) & require("datawizard", quietly = TRUE)) {
  # Single variable is of interest; all others are "fixed" ------------------
  # Factors
  get_datagrid(iris, at = "Species") # Returns all the levels
  get_datagrid(iris, at = "Species = c('setosa', 'versicolor')") # Specify an expression

  # Numeric variables
  get_datagrid(iris, at = "Sepal.Length") # default spread length = 10
  get_datagrid(iris, at = "Sepal.Length", length = 3) # change length
  get_datagrid(iris[2:150, ],
    at = "Sepal.Length",
    factors = "mode", numerics = "median"
  ) # change non-targets fixing
  get_datagrid(iris, at = "Sepal.Length", range = "ci", ci = 0.90) # change min/max of target
  get_datagrid(iris, at = "Sepal.Length = [0, 1]") # Manually change min/max


  # Standardization and unstandardization
  data <- get_datagrid(iris, at = "Sepal.Length", range = "sd", length = 3)
  data$Sepal.Length # It is a named vector (extract names with `names(out$Sepal.Length)`)
  # TODO: uncomment when datawizard > 0.3.1 is out
  # datawizard::standardize(data, select = "Sepal.Length")
  # data <- get_datagrid(iris, at = "Sepal.Length = c(-2, 0, 2)") # Manually specify values
  # data
  # datawizard::unstandardize(data, select = "Sepal.Length")


  # Multiple variables are of interest, creating a combination --------------
  get_datagrid(iris, at = c("Sepal.Length", "Species"), length = 3)
  get_datagrid(iris, at = c("Sepal.Length", "Petal.Length"), length = c(3, 2))
  get_datagrid(iris, at = c(1, 3), length = 3)
  get_datagrid(iris, at = c("Sepal.Length", "Species"), preserve_range = TRUE)
  get_datagrid(iris, at = c("Sepal.Length", "Species"), numerics = 0)
  get_datagrid(iris, at = c("Sepal.Length = 3", "Species"))
  get_datagrid(iris, at = c("Sepal.Length = c(3, 1)", "Species = 'setosa'"))

  # With list-style at-argument
  get_datagrid(iris, at = list(Sepal.Length = c(1, 3), Species = "setosa"))
}

# With models ===============================================================
# Fit a linear regression
model <- lm(Sepal.Length ~ Sepal.Width * Petal.Length, data = iris)
# Get datagrid of predictors
data <- get_datagrid(model, length = c(20, 3), range = c("range", "sd"))
# Add predictions
data$Sepal.Length <- get_predicted(model, data = data)
# Visualize relationships (each color is at -1 SD, Mean, and + 1 SD of Petal.Length)
plot(data$Sepal.Width, data$Sepal.Length,
  col = data$Petal.Length,
  main = "Relationship at -1 SD, Mean, and + 1 SD of Petal.Length"
)
}
\seealso{
\code{\link[=get_predicted]{get_predicted()}}
}
