% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ConnectionWeights.R
\name{ConnectionWeights}
\alias{ConnectionWeights}
\title{Connection weights method}
\description{
This class implements the \emph{Connection weights} method investigated by
Olden et al. (2004), which results in a relevance score for each input
variable. The basic idea is to multiply all path weights for each
possible connection between an input feature and the output node and then
calculate the sum over them. Besides, it is originally a global
interpretation method and independent of the input data. For a neural
network with \eqn{3} hidden layers with weight matrices \eqn{W_1},
\eqn{W_2} and \eqn{W_3}, this method results in a simple matrix
multiplication independent of the activation functions in between:
\deqn{W_1 * W_2 * W_3.}

In this package, we extended this method to a local method inspired by the
method \emph{Gradient\eqn{\times}Input} (see \code{\link{Gradient}}). Hence, the local variant is
simply the point-wise product of the global \emph{Connection weights} method and
the input data. You can use this variant by setting the \code{times_input}
argument to \code{TRUE} and providing input data.

The R6 class can also be initialized using the \code{\link{run_cw}} function
as a helper function so that no prior knowledge of R6 classes is required.
}
\examples{
\dontshow{if (torch::torch_is_installed()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
#----------------------- Example 1: Torch ----------------------------------
library(torch)

# Create nn_sequential model
model <- nn_sequential(
  nn_linear(5, 12),
  nn_relu(),
  nn_linear(12, 1),
  nn_sigmoid()
)

# Create Converter with input names
converter <- Converter$new(model,
  input_dim = c(5),
  input_names = list(c("Car", "Cat", "Dog", "Plane", "Horse"))
)

# You can also use the helper function for the initialization part
converter <- convert(model,
  input_dim = c(5),
  input_names = list(c("Car", "Cat", "Dog", "Plane", "Horse"))
)

# Apply method Connection Weights
cw <- ConnectionWeights$new(converter)

# Again, you can use a helper function `run_cw()` for initializing
cw <- run_cw(converter)

# Print the head of the result as a data.frame
head(get_result(cw, "data.frame"), 5)

# Plot the result
plot(cw)

#----------------------- Example 2: Neuralnet ------------------------------
if (require("neuralnet")) {
  library(neuralnet)
  data(iris)

  # Train a Neural Network
  nn <- neuralnet((Species == "setosa") ~ Petal.Length + Petal.Width,
    iris,
    linear.output = FALSE,
    hidden = c(3, 2), act.fct = "tanh", rep = 1
  )

  # Convert the trained model
  converter <- convert(nn)

  # Apply the Connection Weights method
  cw <- run_cw(converter)

  # Get the result as a torch tensor
  get_result(cw, type = "torch.tensor")

  # Plot the result
  plot(cw)
}
\dontshow{\}) # examplesIf}
\dontshow{if (torch::torch_is_installed()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# ------------------------- Example 3: Keras -------------------------------
if (require("keras") & keras::is_keras_available()) {
  library(keras)

  # Make sure keras is installed properly
  is_keras_available()

  data <- array(rnorm(10 * 32 * 32 * 3), dim = c(10, 32, 32, 3))

  model <- keras_model_sequential()
  model \%>\%
    layer_conv_2d(
      input_shape = c(32, 32, 3), kernel_size = 8, filters = 8,
      activation = "softplus", padding = "valid") \%>\%
    layer_conv_2d(
      kernel_size = 8, filters = 4, activation = "tanh",
      padding = "same") \%>\%
    layer_conv_2d(
      kernel_size = 4, filters = 2, activation = "relu",
      padding = "valid") \%>\%
    layer_flatten() \%>\%
    layer_dense(units = 64, activation = "relu") \%>\%
    layer_dense(units = 16, activation = "relu") \%>\%
    layer_dense(units = 2, activation = "softmax")

  # Convert the model
  converter <- convert(model)

  # Apply the Connection Weights method
  cw <- run_cw(converter)

  # Get the head of the result as a data.frame
  head(get_result(cw, type = "data.frame"), 5)

  # Plot the result for all classes
  plot(cw, output_idx = 1:2)
}
\dontshow{\}) # examplesIf}
\dontshow{if (torch::torch_is_installed() & Sys.getenv("RENDER_PLOTLY", unset = 0) == 1) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
#------------------------- Plotly plots ------------------------------------
if (require("plotly")) {
  # You can also create an interactive plot with plotly.
  # This is a suggested package, so make sure that it is installed
  library(plotly)
  plot(cw, as_plotly = TRUE)
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item J. D. Olden et al. (2004) \emph{An accurate comparison of methods for
quantifying variable importance in artificial neural networks using
simulated data.} Ecological Modelling 178, p. 389–397
}
}
\seealso{
Other methods: 
\code{\link{DeepLift}},
\code{\link{DeepSHAP}},
\code{\link{ExpectedGradient}},
\code{\link{Gradient}},
\code{\link{IntegratedGradient}},
\code{\link{LIME}},
\code{\link{LRP}},
\code{\link{SHAP}},
\code{\link{SmoothGrad}}
}
\concept{methods}
\section{Super class}{
\code{\link[innsight:InterpretingMethod]{innsight::InterpretingMethod}} -> \code{ConnectionWeights}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{times_input}}{(\code{logical(1)})\cr
This logical value indicates whether the results from
the \emph{Connection weights} method were multiplied by the provided input
data or not. Thus, this value specifies whether the original global
variant of the method or the local one was applied. If the value is
\code{TRUE}, then data is provided in the field \code{data}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-ConnectionWeights-new}{\code{ConnectionWeights$new()}}
\item \href{#method-ConnectionWeights-clone}{\code{ConnectionWeights$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="get_result"><a href='../../innsight/html/InterpretingMethod.html#method-InterpretingMethod-get_result'><code>innsight::InterpretingMethod$get_result()</code></a></span></li>
<li><span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="plot"><a href='../../innsight/html/InterpretingMethod.html#method-InterpretingMethod-plot'><code>innsight::InterpretingMethod$plot()</code></a></span></li>
<li><span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="plot_global"><a href='../../innsight/html/InterpretingMethod.html#method-InterpretingMethod-plot_global'><code>innsight::InterpretingMethod$plot_global()</code></a></span></li>
<li><span class="pkg-link" data-pkg="innsight" data-topic="InterpretingMethod" data-id="print"><a href='../../innsight/html/InterpretingMethod.html#method-InterpretingMethod-print'><code>innsight::InterpretingMethod$print()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ConnectionWeights-new"></a>}}
\if{latex}{\out{\hypertarget{method-ConnectionWeights-new}{}}}
\subsection{Method \code{new()}}{
Create a new instance of the class \code{ConnectionWeights}. When
initialized, the method is applied and the results
are stored in the field \code{result}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ConnectionWeights$new(
  converter,
  data = NULL,
  output_idx = NULL,
  output_label = NULL,
  channels_first = TRUE,
  times_input = FALSE,
  verbose = interactive(),
  dtype = "float"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{converter}}{(\code{\link{Converter}})\cr
An instance of the \code{Converter} class that includes the
torch-converted model and some other model-specific attributes. See
\code{\link{Converter}} for details.\cr}

\item{\code{data}}{(\code{\link{array}}, \code{\link{data.frame}}, \code{\link[torch]{torch_tensor}} or \code{list})\cr
The data to which the method is to be applied. These must
have the same format as the input data of the passed model to the
converter object. This means either
\itemize{
\item an \code{array}, \code{data.frame}, \code{torch_tensor} or array-like format of
size \emph{(batch_size, dim_in)}, if e.g.the model has only one input layer, or
\item a \code{list} with the corresponding input data (according to the
upper point) for each of the input layers.
}
This argument is only relevant if
\code{times_input} is \code{TRUE}, otherwise it will be ignored because it is a
locale (i.e. explanation for each data point individually) method only
in this case.\cr}

\item{\code{output_idx}}{(\code{integer}, \code{list} or \code{NULL})\cr
These indices specify the output nodes for which
the method is to be applied. In order to allow models with multiple
output layers, there are the following possibilities to select
the indices of the output nodes in the individual output layers:
\itemize{
\item An \code{integer} vector of indices: If the model has only one output
layer, the values correspond to the indices of the output nodes, e.g.,
\code{c(1,3,4)} for the first, third and fourth output node. If there are
multiple output layers, the indices of the output nodes from the first
output layer are considered.
\item A \code{list} of \code{integer} vectors of indices: If the method is to be
applied to output nodes from different layers, a list can be passed
that specifies the desired indices of the output nodes for each
output layer. Unwanted output layers have the entry \code{NULL} instead of
a vector of indices, e.g., \code{list(NULL, c(1,3))} for the first and
third output node in the second output layer.
\item \code{NULL} (default): The method is applied to all output nodes in
the first output layer but is limited to the first ten as the
calculations become more computationally expensive for more output
nodes.\cr
}}

\item{\code{output_label}}{(\code{character}, \code{factor}, \code{list} or \code{NULL})\cr
These values specify the output nodes for which
the method is to be applied. Only values that were previously passed with
the argument \code{output_names} in the \code{converter} can be used. In order to
allow models with multiple
output layers, there are the following possibilities to select
the names of the output nodes in the individual output layers:
\itemize{
\item A \code{character} vector or \code{factor} of labels: If the model has only one output
layer, the values correspond to the labels of the output nodes named in the
passed \code{Converter} object, e.g.,
\code{c("a", "c", "d")} for the first, third and fourth output node if the
output names are \code{c("a", "b", "c", "d")}. If there are
multiple output layers, the names of the output nodes from the first
output layer are considered.
\item A \code{list} of \code{charactor}/\code{factor} vectors of labels: If the method is to be
applied to output nodes from different layers, a list can be passed
that specifies the desired labels of the output nodes for each
output layer. Unwanted output layers have the entry \code{NULL} instead of
a vector of labels, e.g., \code{list(NULL, c("a", "c"))} for the first and
third output node in the second output layer.
\item \code{NULL} (default): The method is applied to all output nodes in
the first output layer but is limited to the first ten as the
calculations become more computationally expensive for more output
nodes.\cr
}}

\item{\code{channels_first}}{(\code{logical(1)})\cr
The channel position of the given data (argument
\code{data}). If \code{TRUE}, the channel axis is placed at the second position
between the batch size and the rest of the input axes, e.g.,
\code{c(10,3,32,32)} for a batch of ten images with three channels and a
height and width of 32 pixels. Otherwise (\code{FALSE}), the channel axis
is at the last position, i.e., \code{c(10,32,32,3)}. If the data
has no channel axis, use the default value \code{TRUE}.\cr}

\item{\code{times_input}}{(\code{logical(1)})\cr
Multiplies the results with the input features.
This variant turns the global \emph{Connection weights} method into a local
one. Default: \code{FALSE}.\cr}

\item{\code{verbose}}{(\code{logical(1)})\cr
This logical argument determines whether a progress bar is
displayed for the calculation of the method or not. The default value is
the output of the primitive R function \code{\link[=interactive]{interactive()}}.\cr}

\item{\code{dtype}}{(\code{character(1)})\cr
The data type for the calculations. Use
either \code{'float'} for \code{\link[torch]{torch_float}} or \code{'double'} for
\code{\link[torch]{torch_double}}.\cr}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-ConnectionWeights-clone"></a>}}
\if{latex}{\out{\hypertarget{method-ConnectionWeights-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ConnectionWeights$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
