% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rep_sample_n.R
\name{rep_sample_n}
\alias{rep_sample_n}
\alias{rep_slice_sample}
\title{Perform repeated sampling}
\usage{
rep_sample_n(tbl, size, replace = FALSE, reps = 1, prob = NULL)

rep_slice_sample(
  .data,
  n = NULL,
  prop = NULL,
  replace = FALSE,
  weight_by = NULL,
  reps = 1
)
}
\arguments{
\item{tbl, .data}{Data frame of population from which to sample.}

\item{size, n, prop}{\code{size} and \code{n} refer to the sample size of each sample.
The \code{size} argument to \code{rep_sample_n()} is required, while in
\code{rep_slice_sample()} sample size defaults to 1 if not specified. \code{prop}, an
argument to \code{rep_slice_sample()}, refers to the proportion of rows to sample
in each sample, and is rounded down in the case that \code{prop * nrow(.data)} is
not an integer. When using \code{rep_slice_sample()}, please only supply one of
\code{n} or \code{prop}.}

\item{replace}{Should samples be taken with replacement?}

\item{reps}{Number of samples to take.}

\item{prob, weight_by}{A vector of sampling weights for each of the rows in
\code{.data}—must have length equal to \code{nrow(.data)}.}
}
\value{
A tibble of size \code{reps * n} rows corresponding to \code{reps}
samples of size \code{n} from \code{.data}, grouped by \code{replicate}.
}
\description{
These functions extend the functionality of \code{\link[dplyr:sample_n]{dplyr::sample_n()}} and
\code{\link[dplyr:slice]{dplyr::slice_sample()}} by allowing for repeated sampling of data.
This operation is especially helpful while creating sampling
distributions—see the examples below!
}
\details{
\code{rep_sample_n()} and \code{rep_slice_sample()} are designed to behave similar to
their dplyr counterparts. As such, they have at least the following
differences:
\itemize{
\item In case \code{replace = FALSE} having \code{size} bigger than number of data rows in
\code{rep_sample_n()} will give an error. In \code{rep_slice_sample()} having such \code{n}
or \code{prop > 1} will give warning and output sample size will be set to number
of rows in data.
}

Note that the \code{\link[dplyr:sample_n]{dplyr::sample_n()}} function  has been superseded by
\code{\link[dplyr:slice]{dplyr::slice_sample()}}.
}
\examples{
library(dplyr)
library(ggplot2)
library(tibble)

# take 1000 samples of size n = 50, without replacement
slices <- gss \%>\%
  rep_slice_sample(n = 50, reps = 1000)

slices

# compute the proportion of respondents with a college
# degree in each replicate
p_hats <- slices \%>\%
  group_by(replicate) \%>\%
  summarize(prop_college = mean(college == "degree"))

# plot sampling distribution
ggplot(p_hats, aes(x = prop_college)) +
  geom_density() +
  labs(
    x = "p_hat", y = "Number of samples",
    title = "Sampling distribution of p_hat"
  )

# sampling with probability weights. Note probabilities are automatically
# renormalized to sum to 1
df <- tibble(
  id = 1:5,
  letter = factor(c("a", "b", "c", "d", "e"))
)

rep_slice_sample(df, n = 2, reps = 5, weight_by = c(.5, .4, .3, .2, .1))
}
