% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayes.r
\name{bayes}
\alias{bayes}
\title{Bayes model}
\usage{
bayes(
  y,
  M,
  X = NULL,
  R = NULL,
  model = c("BayesCpi", "BayesA", "BayesL", "BSLMM", "BayesR", "BayesB", "BayesC",
    "BayesBpi", "BayesRR"),
  map = NULL,
  Pi = NULL,
  fold = NULL,
  niter = 20000,
  nburn = 12000,
  windsize = NULL,
  windnum = NULL,
  vg = NULL,
  dfvg = NULL,
  s2vg = NULL,
  ve = NULL,
  dfve = NULL,
  s2ve = NULL,
  lambda = 0,
  outfreq = NULL,
  seed = 666666,
  threads = 4,
  verbose = TRUE
)
}
\arguments{
\item{y}{vector of phenotype, use 'NA' for the missings. The number and order of individuals of y, M, X, R should be exactly the same.}

\item{M}{numeric matrix of genotype with individuals in rows and markers in columns, NAs are not allowed.}

\item{X}{(optional) covariate matrix of all individuals, all values should be in digits, characters are not allowed, please use 'model.matrix.lm' function to prepare it.}

\item{R}{(optional) environmental random effects matrix of all individuals, NAs are not allowed for the individuals with phenotypic value.}

\item{model}{bayes model including: "BayesB", "BayesA", "BayesL", "BayesRR", "BayesBpi", "BayesC", "BayesCpi", "BayesR", "BSLMM".
\itemize{
\item "BayesRR": Bayes Ridge Regression, all SNPs have non-zero effects and share the same variance, equals to RRBLUP or GBLUP.
\item "BayesA": all SNPs have non-zero effects, and take different variance which follows an inverse chi-square distribution.
\item "BayesB": only a small proportion of SNPs (1-Pi) have non-zero effects, and take different variance which follows an inverse chi-square distribution.
\item "BayesBpi": the same with "BayesB", but 'Pi' is not fixed.
\item "BayesC": only a small proportion of SNPs (1-Pi) have non-zero effects, and share the same variance.
\item "BayesCpi": the same with "BayesC", but 'Pi' is not fixed.
\item "BayesL": BayesLASSO, all SNPs have non-zero effects, and take different variance which follows an exponential distribution.
\item "BSLMM": all SNPs have non-zero effects, and take the same variance, but a small proportion of SNPs have additional shared variance.
\item "BayesR": only a small proportion of SNPs have non-zero effects, and the SNPs are allocated into different groups, each group has the same variance.
}}

\item{map}{(optional, only for GWAS) the map information of genotype, at least 3 columns are: SNPs, chromosome, physical position.}

\item{Pi}{vector, the proportion of zero effect and non-zero effect SNPs, the first value must be the proportion of non-effect markers.}

\item{fold}{proportion of variance explained for groups of SNPs, the default is c(0, 0.0001, 0.001, 0.01).}

\item{niter}{the number of MCMC iteration.}

\item{nburn}{the number of iterations to be discarded.}

\item{windsize}{window size in bp for GWAS, the default is NULL.}

\item{windnum}{fixed number of SNPs in a window for GWAS, if it is specified, 'windsize' will be invalid, the default is NULL.}

\item{vg}{prior value of genetic variance.}

\item{dfvg}{the number of degrees of freedom for the distribution of genetic variance.}

\item{s2vg}{scale parameter for the distribution of genetic variance.}

\item{ve}{prior value of residual variance.}

\item{dfve}{the number of degrees of freedom for the distribution of residual variance.}

\item{s2ve}{scale parameter for the distribution of residual variance.}

\item{lambda}{value of ridge regression for inverting a matrix.}

\item{outfreq}{frequency of collecting the estimated parameters and printing on console. Note that smaller frequency may have higher accuracy of estimated parameters, but would result in more time and memory for collecting process, on contrary, bigger frequency may have an negative effect on accuracy of estimations.}

\item{seed}{seed for random sample.}

\item{threads}{number of threads used for OpenMP.}

\item{verbose}{whether to print the iteration information on console.}
}
\value{
the function returns a list containing
\describe{
\item{$mu}{the regression intercept}
\item{$pi}{estimated proportion of zero effect and non-zero effect SNPs}
\item{$beta}{estimated coefficients for all covariates}
\item{$r}{estimated environmental random effects}
\item{$Vr}{estimated variance for all environmental random effect}
\item{$Vg}{estimated genetic variance}
\item{$Ve}{estimated residual variance}
\item{$h2}{estimated heritability (h2 = Vg / (Vr + Vg + Ve))}
\item{$alpha}{estimated effect size of all markers}
\item{$g}{genomic estimated breeding value}
\item{$e}{residuals of the model}
\item{$pip}{the frequency for markers to be included in the model during MCMC iteration, known as posterior inclusive probability (PIP)}
\item{$gwas}{WPPA is defined to be the window posterior probability of association, it is estimated by counting the number of MCMC samples in which \deqn{\alpha} is nonzero for at least one SNP in the window}
\item{$MCMCsamples}{the collected samples of posterior estimation for all the above parameters across MCMC iterations}
}
}
\description{
Bayes linear regression model using individual level data
\deqn{y = X \beta + R r + M \alpha + e}
where \eqn{\beta} is a vector of estimated coefficient for covariates, and \eqn{r} is a vector of environmental random effects. \eqn{M} is a matrix of genotype covariate, \eqn{\alpha} is a vector of estimated marker effect size. \eqn{e} is a vector of residuals.
}
\examples{
# Load the example data attached in the package
pheno_file_path = system.file("extdata", "pheno.txt", package = "hibayes")
pheno = read.table(pheno_file_path, header=TRUE)
bfile_path = system.file("extdata", "geno", package = "hibayes")
data = read_plink(bfile_path, out=tempfile())
fam = data$fam
geno = data$geno
map = data$map

# Adjust the order of phenotype by genotype id
geno.id = fam[, 2]
pheno = pheno[match(geno.id, pheno[, 1]), ]

# Add fixed effects, covariates, and random effect
X <- model.matrix.lm(~as.numeric(scale)+as.factor(sex), data=pheno, na.action = "na.pass")
X <- X[, -1] #remove the intercept
# then fit the model as: fit = bayes(..., X=X, R=pheno[,c("group")], ...)

# For GS/GP
fit = bayes(y=pheno[, 2], M=geno, model="BayesR", niter=200, nburn=100)
\donttest{
# For GWAS
fit = bayes(y=pheno[, 2], M=geno, map=map, windsize=1e6, model="BayesCpi")
}

# The standard deviation of unknow parameters can be obtained from the list 'MCMCsamples':
# get the SD of estimated SNP effects for markers
snp_effect_sd = apply(fit$MCMCsamples$alpha, 1, sd)
# get the prediction error variance (PEV) of estimated breeding values
gebv_pev = apply(fit$MCMCsamples$g, 1, var)

}
\references{
Meuwissen, Theo HE, Ben J. Hayes, and Michael E. Goddard. "Prediction of total genetic value using genome-wide dense marker maps." Genetics 157.4 (2001): 1819-1829. \cr
de los Campos, G., Hickey, J. M., Pong-Wong, R., Daetwyler, H. D., and Calus, M. P. (2013). Whole-genome regression and prediction methods applied to plant and animal breeding. Genetics, 193(2), 327-345. \cr
Habier, David, et al. "Extension of the Bayesian alphabet for genomic selection." BMC bioinformatics 12.1 (2011): 1-12. \cr
Yi, Nengjun, and Shizhong Xu. "Bayesian LASSO for quantitative trait loci mapping." Genetics 179.2 (2008): 1045-1055. \cr
Zhou, Xiang, Peter Carbonetto, and Matthew Stephens. "Polygenic modeling with Bayesian sparse linear mixed models." PLoS genetics 9.2 (2013): e1003264. \cr
Moser, Gerhard, et al. "Simultaneous discovery, estimation and prediction analysis of complex traits using a Bayesian mixture model." PLoS genetics 11.4 (2015): e1004969. \cr
}
