/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5ESmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5ESpkg.h"     
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 

hid_t
H5EScreate(void)
{
    H5ES_t *es;                          
    hid_t   ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (es = H5ES__create()))
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTCREATE, H5I_INVALID_HID, "can't create event set");

    
    if ((ret_value = H5I_register(H5I_EVENTSET, es, true)) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTREGISTER, H5I_INVALID_HID, "can't register event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESinsert_request(hid_t es_id, hid_t connector_id, void *request)
{
    H5ES_t           *es;                  
    H5VL_connector_t *connector = NULL;    
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");
    if (NULL == request)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL request pointer");
    if (NULL == (connector = H5I_object_verify(connector_id, H5I_VOL)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a VOL connector ID");

    
    if (H5ES__insert_request(es, connector, request) < 0)
        HGOTO_ERROR(H5E_EVENTSET, H5E_CANTINSERT, FAIL, "can't insert request into event set");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESget_count(hid_t es_id, size_t *count )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");

        
        if (count)
            *count = H5ES__list_count(&es->active);
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESget_op_counter(hid_t es_id, uint64_t *op_counter )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");

        
        if (op_counter)
            *op_counter = es->op_counter;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESget_requests(hid_t es_id, H5_iter_order_t order, hid_t *connector_ids, void **requests, size_t array_len,
                 size_t *count )
{
    H5ES_t *es;                  
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");

    
    if (array_len > 0 && (requests || connector_ids))
        if (H5ES__get_requests(es, order, connector_ids, requests, array_len) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTGET, FAIL, "can't get requests");

    
    if (count)
        *count = H5ES__list_count(&es->active);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESwait(hid_t es_id, uint64_t timeout, size_t *num_in_progress , bool *op_failed )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");
        if (NULL == num_in_progress)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL num_in_progress pointer");
        if (NULL == op_failed)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL op_failed pointer");

        
        if (H5ES__wait(es, timeout, num_in_progress, op_failed) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTWAIT, FAIL, "can't wait on operations");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5EScancel(hid_t es_id, size_t *num_not_canceled , bool *op_failed )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");
        if (NULL == num_not_canceled)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL num_not_canceled pointer");
        if (NULL == op_failed)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL op_failed pointer");

        
        if (H5ES__cancel(es, num_not_canceled, op_failed) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTCANCEL, FAIL, "can't cancel operations");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESget_err_status(hid_t es_id, bool *err_status )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");

        
        if (err_status)
            *err_status = es->err_occurred;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESget_err_count(hid_t es_id, size_t *num_errs )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");

        
        if (num_errs) {
            if (es->err_occurred)
                *num_errs = H5ES__list_count(&es->failed);
            else
                *num_errs = 0;
        } 
    }     

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESget_err_info(hid_t es_id, size_t num_err_info, H5ES_err_info_t err_info[] ,
                 size_t *num_cleared )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");
        if (0 == num_err_info)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "err_info array size is 0");
        if (NULL == err_info)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL err_info array pointer");
        if (NULL == num_cleared)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL errors cleared pointer");

        
        if (H5ES__get_err_info(es, num_err_info, err_info, num_cleared) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTGET, FAIL, "can't retrieve error info for failed operation(s)");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESfree_err_info(size_t num_err_info, H5ES_err_info_t err_info[])
{
    size_t u;                   
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (0 == num_err_info)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "err_info array size is 0");
    if (NULL == err_info)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL err_info array pointer");

    
    for (u = 0; u < num_err_info; u++) {
        H5MM_xfree(err_info[u].api_name);
        H5MM_xfree(err_info[u].api_args);
        H5MM_xfree(err_info[u].app_file_name);
        H5MM_xfree(err_info[u].app_func_name);
        if (H5I_dec_app_ref(err_info[u].err_stack_id) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTDEC, FAIL, "can't close error stack for err_info #%llu", (unsigned long long)u);
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESregister_insert_func(hid_t es_id, H5ES_event_insert_func_t func, void *ctx)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");
        if (NULL == func)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL function callback pointer");

        
        es->ins_func = func;
        es->ins_ctx  = ctx;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESregister_complete_func(hid_t es_id, H5ES_event_complete_func_t func, void *ctx)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        H5ES_t *es; 

        
        if (NULL == (es = H5I_object_verify(es_id, H5I_EVENTSET)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid event set identifier");
        if (NULL == func)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "NULL function callback pointer");

        
        es->comp_func = func;
        es->comp_ctx  = ctx;
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5ESclose(hid_t es_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5ES_NONE != es_id) {
        
        if (H5I_EVENTSET != H5I_get_type(es_id))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an event set");

        
        if (H5I_dec_app_ref(es_id) < 0)
            HGOTO_ERROR(H5E_EVENTSET, H5E_CANTDEC, FAIL, "unable to decrement ref count on event set");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 
