% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_sample.R
\name{gs_sample}
\alias{gs_sample}
\title{GridSample sampling algorithm}
\usage{
gs_sample(population_raster, strata_raster, urban_raster,
  desired_cell_size = NA, cfg_hh_per_stratum, cfg_hh_per_urban,
  cfg_hh_per_rural, max_psu_size = Inf, min_pop_per_cell, cfg_pop_per_psu,
  cfg_sample_rururb = FALSE, cfg_sample_spatial = FALSE,
  cfg_sample_spatial_scale, output_path, sample_name)
}
\arguments{
\item{population_raster}{Raster* layer. Input population raster for PSU creation. Values should be number of people in each pixel.}

\item{strata_raster}{Raster* layer. IRaster that defines the stratum that each pixel lies within. Generally created by rasterizing shapefile of polygons that indicate strata.}

\item{urban_raster}{Raster* layer. IRaster of urbanized areas. where a cell value of 1 indicates urban cells and 0 indicates rural cells.}

\item{desired_cell_size}{numeric. Desired cell size (in square kilometers) for output raster of PSUs. Defaults to NA, which yields an output raster at the same resolution as population_raster.}

\item{cfg_hh_per_stratum}{numeric. Target household sample size per stratum. In a non-stratified sample, this is the total sample size of households. In a stratified sample, this is the household sample size per stratum.}

\item{cfg_hh_per_urban}{numeric. Number of households expected to be selected per urban PSU during survey fieldwork.}

\item{cfg_hh_per_rural}{numeric. Number of households expected to be selected per rural PSU during survey fieldwork.}

\item{max_psu_size}{numeric. Maximum allowed geographic size of a given PSU in square kilometers. Defaults to infinity.}

\item{min_pop_per_cell}{numeric. Minimum population in a raster cell required for it to be considered for sampling. Cells with less than this value will be excluded from the sample. Defaults to 0, therefore including all cells.}

\item{cfg_pop_per_psu}{numeric. Target minimum population per PSU.}

\item{cfg_sample_rururb}{logical. Oversample rural/urban areas if one domain does not meet the target sample size per stratum? Default is \code{FALSE}.}

\item{cfg_sample_spatial}{logical. Oversample to ensure that at least one PSU is found within a larger grid with cell size defined by user? Default is \code{FALSE}.}

\item{cfg_sample_spatial_scale}{If \code{cfg_sample_spatial == TRUE}, this defines the cell size in the larger grid where the algorithm will ensure at least one PSU is located in each larger grid cell. Defined in square kilometers.}

\item{output_path}{character. Output folder name.}

\item{sample_name}{character. Name of output PSU raster and shapefile.}
}
\value{
Shapefile of household survey primary sampling unit (PSU) boundaries
}
\description{
The \code{gs_sample} algorithm creates primary sampling units (PSUs) for multi-stage cluster household surveys based on gridded population data. Typical complex survey design is supported with input of, a raster of urbanized areas, and a raster of study strata. Each of these rasters need to be in an identical projection and have an identical grid resolution.
}
\details{
A number of sampling features are optional. Oversampling in urban/rural areas, oversampling to be spatially representative, and stratification are not required. At a minimum, the user generates a simple random sample of PSUs in a study area by inputting a \code{population_raster}, defining the study area boundary as one stratum with strata_raster, defining the output shapefile parameters \code{output_path} and \code{sample_name}, and configuring the parameters \code{cfg_hh_per_stratum}, \code{cfg_hh_per_urban}, \code{cfg_hh_per_rural}, and \code{cfg_pop_per_psu}.  See the "Stratification", "Urban/rural domains", "Spatial sampling", and "PSU size and framework" sections for additional information.
}
\section{Stratification}{

To stratify the sample, define strata geographic boundaries with \code{strata_raster}, and specify the sample size per strata with \code{cfg_hh_per_stratum}. For example, if a national sample will have 10,000 households from 5 provinces, then \code{cfg_hh_per_stratum = 2000}. The parameter \code{cfg_hh_per_stratum} is the minimum sample size to generate representative population statistics. In some surveys, strata are represent urban/rural populations within administrative units. If this is the case, then \code{strata_raster} should include the boundaries of urban and rural sampling areas within each administrative area, and \code{cfg_hh_per_stratum} should reflect the correct sample size per stratum - for example, a national sample of 10,000 households from each urban and rural areas in 5 provinces would have \code{cfg_hh_per_stratum = 1000}.
}

\section{Urban/rural domains}{

If urban/rural populations are not part of the stratification scheme, then they are often treated as a sub-domain. Sub-domains represent important sub-populations for which representative statistics are generated from the survey data, and thus each sub-domain should meet the minimum sample size set for each stratum. If either the urban/rural sub-domain does not include enough households to generate population statistics, then the sub-domain is oversampled. To implement this step with \code{gs_sample}, set \code{cfg_sample_rururb = 1}. In practice, rural areas are often more difficult and expensive to visit, and thus a greater number of households might be sampled from rural PSUs than urban PSUs. This is why the user may specify different numbers of households to be sampled from each urban PSUs (\code{cfg_hh_per_urban}) and rural PSUs (\code{cfg_hh_per_rural}); if the same number of households will be sampled from all PSUs, then configure both of these parameters with the same value. Note, the number of PSUs that will be generated in each stratum is \code{cfg_hh_per_stratum} divided by some number between \code{cfg_hh_per_urban} and \code{cfg_hh_per_rural}.
}

\section{Spatial sampling}{

To select a sample that is both representative of the population and of space, set \code{cfg_sample_spatial = 1} and specify \code{cfg_sample_spatial_scale}, the spatial scale at which the sample should be representative. The spatial scale should be meaningful; for example, it will facilitate small area estimates with limited statistical error for the administrative units below the administrative units used to stratify the sample. Determining an appropriate spatial scale might take trial and error. If the study area has large regions of sparse population, a typical non-spatially representative sample will follow a distribution similar to the population and have large areas without a PSU. In this case, the user might need to increase the spatial resolution of the sample, or force the algorithm to generate more PSUs in each stratum by increasing \code{cfg_hh_per_stratum} and/or reducing \code{cfg_hh_per_urban} and \code{cfg_hh_per_rural}.
}

\section{PSU size and fieldwork}{

Three additional parameters can be configured to deal with idiosyncrasies of gridded population data and improve feasibility of fieldwork. The user can set a maximum geographic size of PSU in square kilometers, \code{max_psu_size}. We recommend choosing a size that can feasibly be visited on foot during one day. The user might also specify which cells are included in the sample frame with \code{min_pop_per_cell}. Selection of a sensible value is highly dependent on the specific gridded population dataset being used, and the scale of the input data (eg 200m grid cells). Finally, the cell size of the output raster can be specified with \code{desired_cell_size}, which can be modified to account for the expected accuracy of the input gridded population datasets.
}
\examples{
require(raster)
poprast <- raster(ncols=50,nrows=50,xmx=10,xmn=9,ymn=9,ymx=10,
crs=CRS("+proj=longlat +datum=WGS84 +no_defs +ellps=WGS84 +towgs84=0,0,0"),
vals=runif(2500,0,1))
stratarast <- raster(ncols=50,nrows=50,xmx=10,xmn=9,ymn=9,ymx=10,
crs=CRS("+proj=longlat +datum=WGS84 +no_defs +ellps=WGS84 +towgs84=0,0,0"),
vals=c(rep(1,times=1250),rep(2,times=1250)))
urbanrast <- poprast > 0.9
gs_sample(population_raster = poprast, 
          strata_raster = stratarast, 
          urban_raster = urbanrast, 
          desired_cell_size = NA,
          cfg_hh_per_stratum = 20,
          cfg_hh_per_urban = 2, 
          cfg_hh_per_rural = 2, 
          min_pop_per_cell = 0.01,
          cfg_pop_per_psu = 10,
          cfg_sample_rururb = FALSE, 
          cfg_sample_spatial = FALSE, 
          cfg_sample_spatial_scale = 100,
          output_path=tempdir(),
          sample_name="Example")
}

