% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sils.R
\name{sils}
\alias{sils}
\title{Structural Iterated Least Squares (SILS)}
\usage{
sils(dependent_variable, distance, additional_regressors = NULL,
  income_origin, income_destination, code_origin, code_destination,
  maxloop = 100, decimal_places = 4, robust = FALSE,
  verbose = FALSE, data, ...)
}
\arguments{
\item{dependent_variable}{(Type: character) name of the dependent variable. This dependent variable is 
divided by the product of unilateral incomes such (i.e. \code{income_origin} and \code{income_destination}) 
and logged afterwards.}

\item{distance}{(Type: character) name of the distance variable that should be taken as the key independent variable 
in the estimation. The distance is logged automatically when the function is executed.}

\item{additional_regressors}{(Type: character) names of the additional regressors to include in the model (e.g. a dummy
variable to indicate contiguity). Unilateral metric variables such as GDP should be inserted via the arguments 
\code{income_origin} and \code{income_destination}. As country specific effects are subdued due to demeaning, no further unilateral variables apart from incomes can be added.

Write this argument as \code{c(contiguity, common currency, ...)}. By default this is set to \code{NULL}.}

\item{income_origin}{(Type: character) origin income variable (e.g. GDP) in the dataset.}

\item{income_destination}{(Type: character) destination income variable (e.g. GDP) in the dataset.}

\item{code_origin}{(Type: character) country of origin variable (e.g. ISO-3 country codes). The variables are grouped 
using this parameter.}

\item{code_destination}{(Type: character) country of destination variable (e.g. country ISO-3 codes). The variables are 
grouped using this parameter.}

\item{maxloop}{(Type: numeric) maximum number of outer loop iterations.
The default is set to 100. There will be a warning if the iterations
did not converge.}

\item{decimal_places}{(Type: numeric) number of decimal places that should not change after a new
iteration for the estimation to stop. The default is set to 4.}

\item{robust}{(Type: logical) whether robust fitting should be used. By default this is set to \code{FALSE}.}

\item{verbose}{(Type: logical) determines whether the estimated coefficients
of each iteration should be printed in the console. The default is set
to \code{FALSE}.}

\item{data}{(Type: data.frame) the dataset to be used.}

\item{...}{Additional arguments to be passed to the function.}
}
\value{
The function returns the summary of the estimated gravity model as an
\code{\link[stats]{lm}}-object. It furthermore returns the resulting coefficients for each
iteration.
}
\description{
\code{sils} estimates gravity models via
Structural Iterated Least Squares and an explicit inclusion
of the Multilateral Resistance terms.
}
\details{
\code{sils} is an estimation method for gravity models
developed by \insertCite{Head2014;textual}{gravity}.

The function \code{sils} utilizes the relationship between the Multilateral
Resistance terms and the transaction costs. The parameters are estimated by
an iterative procedure. The function executes loops until the parameters
stop changing significantly.

\code{sils} is designed to be consistent with the Stata code provided at
\href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook}
when choosing robust estimation.

As, to our knowledge at the moment, there is no explicit literature covering
the estimation of a gravity equation by \code{sils} using panel data,
and we do not recommend to apply this method in this case.
}
\examples{
# Example for CRAN checks:
# Executable in < 5 sec
library(dplyr)
data("gravity_no_zeros")

# Choose 5 countries for testing
countries_chosen <- c("AUS", "CHN", "GBR", "BRA", "CAN")
grav_small <- filter(gravity_no_zeros, iso_o \%in\% countries_chosen)

fit <- sils(
  dependent_variable = "flow",
  distance = "distw",
  additional_regressors = "rta",
  income_origin = "gdp_o",
  income_destination = "gdp_d",
  code_origin = "iso_o",
  code_destination = "iso_d",
  maxloop = 50,
  dec_places = 3,
  robust = FALSE,
  verbose = FALSE,
  data = grav_small
)

}
\references{
For more information on gravity models, theoretical foundations and
estimation methods in general see

\insertRef{Anderson1979}{gravity}

\insertRef{Anderson2001}{gravity}

\insertRef{Anderson2010}{gravity}

\insertRef{Baier2009}{gravity}

\insertRef{Baier2010}{gravity}

\insertRef{Feenstra2002}{gravity}

\insertRef{Head2010}{gravity}

\insertRef{Head2014}{gravity}

\insertRef{Santos2006}{gravity}

and the citations therein.

See \href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook} for gravity datasets and Stata code for estimating gravity models.

For estimating gravity equations using panel data see

\insertRef{Egger2003}{gravity}

\insertRef{Gomez-Herrera2013}{gravity}

and the references therein.
}
\seealso{
\code{\link[stats]{lm}}, \code{\link[lmtest]{coeftest}},
\code{\link[sandwich]{vcovHC}}
}
