% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fpc.R
\name{fpc}
\alias{fpc}
\title{Computation of functional principal components}
\usage{
fpc(X_fdata, n_fpc = 3, centered = FALSE, int_rule = "trapezoid",
  equispaced = FALSE, verbose = FALSE)
}
\arguments{
\item{X_fdata}{sample of functional data as an
\code{\link[fda.usc]{fdata}} object of length \code{n}.}

\item{n_fpc}{number of FPC to be computed. If \code{n_fpc > n}, \code{n_fpc}
is set to \code{n}. Defaults to \code{3}.}

\item{centered}{flag to indicate if \code{X_fdata} is centered or not.
Defaults to \code{FALSE}.}

\item{int_rule}{quadrature rule for approximating the definite
unidimensional integral: trapezoidal rule (\code{int_rule = "trapezoid"})
and extended Simpson rule (\code{int_rule = "Simpson"}) are available.
Defaults to \code{"trapezoid"}.}

\item{equispaced}{flag to indicate if \code{X_fdata$data} is valued in
an equispaced grid or not. Defaults to \code{FALSE}.}

\item{verbose}{whether to show or not information about the \code{fpc}
procedure. Defaults to \code{FALSE}.}
}
\value{
An \code{"fpc"} object containing the following elements:
\item{d}{standard deviations of the FPC (i.e., square roots of
eigenvalues of the empirical autocovariance estimator).}
\item{rotation}{orthonormal eigenfunctions (loadings or functional
principal components), as an \code{\link[fda.usc]{fdata}} class object.}
\item{scores}{rotated samples: inner products.
between \code{X_fdata} and eigenfunctions in\cr \code{rotation}.}
\item{l}{vector of index of FPC.}
\item{equispaced}{\code{equispaced} flag.}
}
\description{
Computation of Functional Principal Components (FPC) for
equispaced and non equispaced functional data.
}
\details{
The FPC are obtained by performing the single value decomposition
\deqn{\mathbf{X} \mathbf{W}^{1/2} =
\mathbf{U} \mathbf{D} (\mathbf{V}' \mathbf{W}^{1/2})}{
X W^{1/2} = U D (V' W^{1/2})}
where \eqn{\mathbf{X}}{X} is the matrix of discretized functional data,
\eqn{\mathbf{W}}{W} is a diagonal matrix of weights (computed by
\code{w_integral1D} according to \code{int_rule}), \eqn{\mathbf{D}}{D}
is the diagonal matrix with singular values (standard deviations of FPC),
\eqn{\mathbf{U}}{U} is a matrix whose columns contain the left singular
vectors, and \eqn{\mathbf{V}}{V} is a matrix whose columns contain the
right singular vectors (FPC).
}
\examples{
## Computing FPC for equispaced data

# Sample data
X_fdata1 <- r_ou(n = 200, t = seq(2, 4, l = 201))

# FPC with trapezoid rule
X_fpc1 <- fpc(X_fdata = X_fdata1, n_fpc = 50, equispaced = TRUE,
              int_rule = "trapezoid")

# FPC with Simpsons's rule
X_fpc2 <- fpc(X_fdata = X_fdata1, n_fpc = 50, equispaced = TRUE,
               int_rule = "Simpson")

# Check if FPC are orthonormal
norms1 <- rep(0, length(X_fpc1$l))
for (i in X_fpc1$l) {

  norms1[i] <- integral1D(fx = X_fpc1$rotation$data[i, ]^2,
                          t = X_fdata1$argvals)

}
norms2 <- rep(0, length(X_fpc2$l))
for (i in X_fpc2$l) {

  norms2[i] <- integral1D(fx = X_fpc2$rotation$data[i, ]^2,
                          t = X_fdata1$argvals)

}

## Computing FPC for non equispaced data

# Sample data
X_fdata2 <- r_ou(n = 200, t = c(seq(0, 0.5, l = 201), seq(0.51, 1, l = 301)))

# FPC with trapezoid rule
X_fpc3 <- fpc(X_fdata = X_fdata2, n_fpc = 5, int_rule = "trapezoid",
              equispaced = FALSE)

# Check if FPC are orthonormal
norms3 <- rep(0, length(X_fpc3$l))
for (i in X_fpc3$l) {

  norms3[i] <- integral1D(fx = X_fpc3$rotation$data[i, ]^2,
                          t = X_fdata2$argvals)

}
\donttest{
## Efficiency comparisons

# fpc() vs. fda.usc::fdata2pc()
data(phoneme, package = "fda.usc")
mlearn <- phoneme$learn[1:10, ]
res1 <- fda.usc::fdata2pc(mlearn, ncomp = 3)
res2 <- fpc(X_fdata = mlearn, n_fpc = 3)
plot(res1$x[, 1:3], col = 1)
points(res2$scores, col = 2)

microbenchmark::microbenchmark(fda.usc::fdata2pc(mlearn, ncomp = 3),
                               fpc(X_fdata = mlearn, n_fpc = 3), times = 1e3,
                               control = list(warmup = 20))
}
}
\references{
Jolliffe, I. T. (2002). Principal Component Analysis. Springer-Verlag,
New York.
}
\author{
Javier Álvarez-Liébana and Gonzalo Álvarez-Pérez.
}
