findGlobals_conservative <- function(expr, envir, ...) {
  fun <- asFunction(expr, envir=envir, ...)
  names <- codetools::findGlobals(fun, merge=TRUE)
  names
}

findGlobals_liberal <- function(expr, envir, ...) {
  requireNamespace("codetools")

  vars <- new.env(hash=TRUE, parent=emptyenv())
  funs <- new.env(hash=TRUE, parent=emptyenv())

  enter <- function(type, v, e, w) {
    if (type == "function")
      assign(v, value=TRUE, envir=funs)
    else
      assign(v, value=TRUE, envir=vars)
  }

  fun <- asFunction(expr, envir=envir, ...)
  w <- codetools::makeUsageCollector(fun, enterGlobal=enter, name="<anonymous>")
  codetools::walkCode(expr, w)

  fnames <- ls(funs, all.names = TRUE)
  vnames <- ls(vars, all.names = TRUE)
  sort(unique(c(fnames, vnames)))
}


#' @export
findGlobals <- function(expr, envir=parent.frame(), ..., tweak=NULL, dotdotdot=c("warning", "error", "return", "ignore"), method=c("conservative", "liberal"), substitute=FALSE, unlist=TRUE) {
  method <- match.arg(method)
  dotdotdot <- match.arg(dotdotdot)

  if (substitute) expr <- substitute(expr)

  if (is.list(expr)) {
    globals <- lapply(expr, FUN=findGlobals, envir=envir, ..., tweak=tweak, dotdotdot=dotdotdot, unlist=FALSE)
    if (unlist) {
      needsDotdotdot <- FALSE
      for (kk in seq_along(globals)) {
        s <- globals[[kk]]
        n <- length(s)
        if (identical(s[n], "...")) {
          needsDotdotdot <- TRUE
          s <- s[-n]
          globals[[kk]] <- s
        }
      }
      globals <- unlist(globals, use.names=TRUE)
      globals <- sort(unique(globals))
      if (needsDotdotdot) globals <- c(globals, "...")
    }
    return(globals)
  }

  if (is.function(tweak)) expr <- tweak(expr)

  if (method == "liberal") {
    findGlobalsT <- findGlobals_liberal
  } else {
    findGlobalsT <- findGlobals_conservative
  }

  ## Is there a need for global '...' variables?
  needsDotdotdot <- FALSE
  globals <- withCallingHandlers({
    oopts <- options(warn=0L)
    on.exit(options(oopts))
    findGlobalsT(expr, envir=envir)
  }, warning=function(w) {
    ## Warned about '...'?
    pattern <- "... may be used in an incorrect context"
    if (grepl(pattern, w$message, fixed=TRUE)) {
      needsDotdotdot <<- TRUE
      if (dotdotdot == "return") {
        ## Consume / muffle warning
        invokeRestart("muffleWarning")
      } else if (dotdotdot == "ignore") {
        needsDotdotdot <<- FALSE
        ## Consume / muffle warning
        invokeRestart("muffleWarning")
      } else if (dotdotdot == "error") {
        e <- simpleError(w$message, w$call)
        stop(e)
      }
    }
  })

  if (needsDotdotdot) globals <- c(globals, "...")

  globals
}
