% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lkjcorr_marginal.R
\name{marginalize_lkjcorr}
\alias{marginalize_lkjcorr}
\title{Turn spec for LKJ distribution into spec for marginal LKJ distribution}
\usage{
marginalize_lkjcorr(data, K, predicate = NULL, dist = ".dist", args = ".args")
}
\arguments{
\item{data}{A data frame containing a column with distribution names (\code{".dist"} by default)
and a list column of distribution arguments (\code{".args"} by default), such as output by
\code{\link[=parse_dist]{parse_dist()}}.}

\item{K}{Dimension of the correlation matrix. Must be greater than or equal to 2.}

\item{predicate}{a bare expression for selecting the rows of \code{data} to modify. This is useful
if \code{data} contains more than one row with an LKJ prior in it and you only want to modify some
of the distributions; if this is the case, give row a predicate expression (such as you might supply
to \code{\link[dplyr:filter]{dplyr::filter()}}) that evaluates to \code{TRUE} on the rows you want to modify.
If \code{NULL} (the default), all \code{lkjcorr} distributions in \code{data} are modified.}

\item{dist}{The name of the column containing distribution names. See \code{\link[=parse_dist]{parse_dist()}}.}

\item{args}{The name of the column containing distribution arguments. See \code{\link[=parse_dist]{parse_dist()}}.}
}
\value{
A data frame of the same size and column names as the input, with the \code{dist} and \code{args}
columns modified on rows where \code{dist == "lkjcorr"} such that they represent a
marginal LKJ correlation distribution with name \code{lkjcorr_marginal} and \code{args} having
\code{K} equal to the input value of \code{K}.
}
\description{
Turns specs for an LKJ correlation matrix distribution as returned by
\code{\link[=parse_dist]{parse_dist()}} into specs for the marginal distribution of
a single cell in an LKJ-distributed correlation matrix (i.e., \code{\link[=lkjcorr_marginal]{lkjcorr_marginal()}}).
Useful for visualizing prior correlations from LKJ distributions.
}
\details{
The LKJ(eta) prior on a correlation matrix induces a marginal prior on each correlation
in the matrix that depends on both the value of \code{eta} \emph{and} \code{K},the dimension
of the \eqn{K \times K}{KxK} correlation matrix. Thus to visualize the marginal prior
on the correlations, it is necessary to specify the value of \code{K}, which depends
on what your model specification looks like.

Given a data frame representing parsed distribution specifications (such
as returned by \code{\link[=parse_dist]{parse_dist()}}), this function updates any rows with \code{.dist == "lkjcorr"}
so that the first argument to the distribution is equal to the specified dimension
of the correlation matrix (\code{K}) and changes the distribution name to \code{"lkjcorr_marginal"},
allowing the distribution to be easily visualized using the \code{\link[=stat_dist_slabinterval]{stat_dist_slabinterval()}}
family of ggplot2 stats.
}
\examples{

library(dplyr)
library(ggplot2)

# Say we have an LKJ(3) prior on a 2x2 correlation matrix. We can visualize
# its marginal distribution as follows...
data.frame(prior = "lkjcorr(3)") \%>\%
  parse_dist(prior) \%>\%
  marginalize_lkjcorr(K = 2) \%>\%
  ggplot(aes(y = prior, dist = .dist, args = .args)) +
  stat_dist_halfeye() +
  xlim(-1, 1) +
  xlab("Marginal correlation for LKJ(3) prior on 2x2 correlation matrix")

# Say our prior list has multiple LKJ priors on correlation matrices
# of different sizes, we can supply a predicate expression to select
# only those rows we want to modify
data.frame(coef = c("a", "b"), prior = "lkjcorr(3)") \%>\%
  parse_dist(prior) \%>\%
  marginalize_lkjcorr(K = 2, coef == "a") \%>\%
  marginalize_lkjcorr(K = 4, coef == "b")

}
\seealso{
\code{\link[=parse_dist]{parse_dist()}}, \code{\link[=lkjcorr_marginal]{lkjcorr_marginal()}}
}
