#' Download official data of urbanized areas in Brazil as an sf object.
#'
#' This function reads the official data on the urban footprint of Brazilian cities
#' in the years 2005 and 2015. Orignal data were generated by Institute of Geography
#' and Statistics (IBGE)  For more information about the methodology, see deails at
#' https://biblioteca.ibge.gov.br/visualizacao/livros/liv100639.pdf
#'
#'
#' @param year A year number in YYYY format (defaults to 2015)
#' @param simplified Logic FALSE or TRUE, indicating whether the function returns the
#'  data set with 'original' resolution or a data set with 'simplified' borders (Defaults to TRUE).
#'  For spatial analysis and statistics users should set simplified = FALSE. Borders have been
#'  simplified by removing vertices of borders using st_simplify{sf} preserving topology with a dTolerance of 100.
#' @param showProgress Logical. Defaults to (TRUE) display progress bar
#'
#' @export
#' @examples \dontrun{
#'
#' library(geobr)
#'
#' # Read urban footprint of Brazilian cities in an specific year
#'   d <- read_urban_area(year=2005)
#'
#' }
#'
#'
read_urban_area <- function(year=2015, simplified=TRUE, showProgress=TRUE){

  # Get metadata with data url addresses
  temp_meta <- select_metadata(geography="urban_area", year=year, simplified=simplified)

  # list paths of files to download
  file_url <- as.character(temp_meta$download_path)

  # download files
  temp_sf <- download_gpkg(file_url, progress_bar = showProgress)
  return(temp_sf)

}
