% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/display.R
\name{plot_raster}
\alias{plot_raster}
\title{Display raster data}
\usage{
plot_raster(
  data,
  xsize = NULL,
  ysize = NULL,
  nbands = 1,
  max_pixels = 2.5e+07,
  col_tbl = NULL,
  maxColorValue = 1,
  normalize = TRUE,
  minmax_def = NULL,
  minmax_pct_cut = NULL,
  col_map_fn = NULL,
  xlim = NULL,
  ylim = NULL,
  interpolate = TRUE,
  asp = 1,
  axes = TRUE,
  main = "",
  xlab = "x",
  ylab = "y",
  xaxs = "i",
  yaxs = "i",
  legend = FALSE,
  digits = 2,
  na_col = rgb(0, 0, 0, 0),
  ...
)
}
\arguments{
\item{data}{Either a \code{GDALRaster} object from which data will be read, or
a numeric vector of pixel values arranged in left to right, top to
bottom order.}

\item{xsize}{The number of pixels along the x dimension in \code{data}. If \code{data}
is a \code{GDALRaster} object, specifies the size at which the raster will be
read (used for argument \code{out_xsize} in \code{GDALRaster$read()}). By default,
the entire raster will be read at full resolution.}

\item{ysize}{The number of pixels along the y dimension in \code{data}. If \code{data}
is a \code{GDALRaster} object, specifies the size at which the raster will be
read (used for argument \code{out_ysize} in \code{GDALRaster$read()}). By default,
the entire raster will be read at full resolution.}

\item{nbands}{The number of bands in \code{data}. Must be either 1 (grayscale) or
3 (RGB). For RGB, \code{data} are interleaved by band.}

\item{max_pixels}{The maximum number of pixels that the function will
attempt to display (per band). An error is raised if \code{(xsize * ysize)}
exceeds this value. Setting to \code{NULL} turns off this check.}

\item{col_tbl}{A color table as a matrix or data frame with four or five
columns. Column 1 contains the numeric pixel values. Columns 2:4 contain
the intensities of the red, green and blue primaries (\code{0:1} by default,
or use integer \code{0:255} by setting \code{maxColorValue = 255}).
An optional column 5 may contain alpha transparency values, \code{0} for fully
transparent to \code{1} (or \code{maxColorValue}) for opaque (the default if column 5
is missing). If \code{data} is a \code{GDALRaster} object, a built-in color table will
be used automatically if one exists in the dataset.}

\item{maxColorValue}{A number giving the maximum of the color values range
in \code{col_tbl} (see above). The default is \code{1}.}

\item{normalize}{Logical. \code{TRUE} to rescale pixel values so that their
range is \verb{[0,1]}, normalized to the full range of the pixel data by default
(\code{min(data)}, \code{max(data)}, per band). Ignored if \code{col_tbl} is used.
Set \code{normalize} to \code{FALSE} if a color map function is used that
operates on raw pixel values (see \code{col_map_fn} below).}

\item{minmax_def}{Normalize to user-defined min/max values (in terms of
the pixel data, per band). For single-band grayscale, a numeric vector of
length two containing min, max. For 3-band RGB, a numeric vector of length
six containing b1_min, b2_min, b3_min, b1_max, b2_max, b3_max.}

\item{minmax_pct_cut}{Normalize to a truncated range of the pixel data using
percentile cutoffs (removes outliers). A numeric vector of length two giving
the percentiles to use (e.g., \code{c(2, 98)}). Applied per band. Ignored if
\code{minmax_def} is used.}

\item{col_map_fn}{An optional color map function (default is
\code{grDevices::gray} for single-band data or \code{grDevices::rgb} for 3-band).
Ignored if \code{col_tbl} is used. Set \code{normalize} to \code{FALSE} if using a color
map function that operates on raw pixel values.}

\item{xlim}{Numeric vector of length two giving the x coordinate range.
If \code{data} is a \code{GDALRaster} object, the default is the raster xmin, xmax in
georeferenced coordinates, otherwise the default uses pixel/line
coordinates (\code{c(0, xsize)}).}

\item{ylim}{Numeric vector of length two giving the y coordinate range.
If \code{data} is a \code{GDALRaster} object, the default is the raster ymin, ymax in
georeferenced coordinates, otherwise the default uses pixel/line
coordinates (\code{c(ysize, 0)}).}

\item{interpolate}{Logical indicating whether to apply linear interpolation
to the image when drawing (default \code{TRUE}).}

\item{asp}{Numeric. The aspect ratio y/x (see \code{?plot.window}).}

\item{axes}{Logical. \code{TRUE} to draw axes (the default).}

\item{main}{The main title (on top).}

\item{xlab}{Title for the x axis (see \code{?title}).}

\item{ylab}{Title for the y axis (see \code{?title}).}

\item{xaxs}{The style of axis interval calculation to be used for the x axis
(see \code{?par}).}

\item{yaxs}{The style of axis interval calculation to be used for the y axis
(see \code{?par}).}

\item{legend}{Logical indicating whether to include a legend on the plot.
Currently, legends are only supported for continuous data. A color table
will be used if one is specified or the raster has a built-in color table,
otherwise the value for \code{col_map_fn} will be used.}

\item{digits}{The number of digits to display after the decimal point in
the legend labels when raster data are floating point.}

\item{na_col}{Color to use for \code{NA} as a 7- or 9-character hexadecimal code.
The default is transparent (\code{"#00000000"}, the return value of
\code{rgb(0,0,0,0)}).}

\item{...}{Other parameters to be passed to \code{plot.default()}.}
}
\description{
\code{plot_raster()} displays raster data using base \code{graphics}.
}
\note{
\code{plot_raster()} uses the function \code{graphics::rasterImage()} for plotting
which is not supported on some devices (see \code{?rasterImage}).

If \code{data} is an object of class \code{GDALRaster}, then \code{plot_raster()} will
attempt to read the entire raster into memory by default (unless the number
of pixels per band would exceed \code{max_pixels}).
A reduced resolution overview can be read by setting \code{xsize}, \code{ysize}
smaller than the raster size on disk.
(If \code{data} is instead specified as a vector of pixel values, a reduced
resolution overview would be read by setting \code{out_xsize} and \code{out_ysize}
smaller than the raster region defined by \code{xsize}, \code{ysize} in a call to
\code{GDALRaster$read()}).
The GDAL_RASTERIO_RESAMPLING configuration option can be
defined to override the default resampling (NEAREST) to one of BILINEAR,
CUBIC, CUBICSPLINE, LANCZOS, AVERAGE or MODE, for example:
\preformatted{
set_config_option("GDAL_RASTERIO_RESAMPLING", "BILINEAR")
}
}
\examples{
## Elevation
elev_file <- system.file("extdata/storml_elev.tif", package="gdalraster")
ds <- new(GDALRaster, elev_file)

# grayscale
plot_raster(ds, legend=TRUE, main="Storm Lake elevation (m)")

# color ramp from user-defined palette
elev_pal <- c("#00A60E","#63C600","#E6E600","#E9BD3B",
              "#ECB176","#EFC2B3","#F2F2F2")
ramp <- scales::colour_ramp(elev_pal, alpha=FALSE)
plot_raster(ds, col_map_fn=ramp, legend=TRUE,
            main="Storm Lake elevation (m)")

ds$close()

## Landsat band combination
b4_file <- system.file("extdata/sr_b4_20200829.tif", package="gdalraster")
b5_file <- system.file("extdata/sr_b5_20200829.tif", package="gdalraster")
b6_file <- system.file("extdata/sr_b6_20200829.tif", package="gdalraster")
band_files <- c(b6_file, b5_file, b4_file)

r <- vector("integer")
for (f in band_files) {
  ds <- new(GDALRaster, f)
  dm <- ds$dim()
  r <- c(r, read_ds(ds))
  ds$close()
}

plot_raster(r, xsize=dm[1], ysize=dm[2], nbands=3,
            main="Landsat 6-5-4 (vegetative analysis)")

## LANDFIRE Existing Vegetation Cover (EVC) with color map
evc_file <- system.file("extdata/storml_evc.tif", package="gdalraster")

# colors from the CSV attribute table distributed by LANDFIRE
evc_csv <- system.file("extdata/LF20_EVC_220.csv", package="gdalraster")
vat <- read.csv(evc_csv)
head(vat)
vat <- vat[,c(1,6:8)]

ds <- new(GDALRaster, evc_file)
plot_raster(ds, col_tbl=vat, interpolate=FALSE,
            main="Storm Lake LANDFIRE EVC")

ds$close()
}
\seealso{
\code{\link[=GDALRaster]{GDALRaster$read()}}, \code{\link[=read_ds]{read_ds()}}, \code{\link[=set_config_option]{set_config_option()}}
}
