% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sync_test.R
\name{sync_test}
\alias{sync_test}
\title{Time Series Trend Synchronicity Test}
\usage{
sync_test(
  formula,
  B = 1000,
  Window = NULL,
  q = NULL,
  j = NULL,
  ar.order = NULL,
  ar.method = "HVK",
  BIC = TRUE
)
}
\arguments{
\item{formula}{an object of class "\code{\link[stats]{formula}}",
specifying the form of the common parametric time trend to be tested
in a \eqn{T} by \eqn{N} matrix of time series
(time series in columns). Variable \eqn{t} should be used to specify the form of
the trend, where \eqn{t} is specified within the function as a regular sequence
on the interval (0,1]. See `Examples'.}

\item{B}{number of bootstrap simulations to obtain empirical critical values.
Default is 1000.}

\item{Window}{scalar or \eqn{N}-vector with lengths of the local windows (factors).
If only one value is set, the same \code{Window} is applied to each time series.
An \eqn{N}-vector gives a specific window for each time series.
If \code{Window} is not specified, an automatic algorithm for optimal
window selection is applied as a default option (see `Details').}

\item{q}{scalar from 0 to 1 to define the set of possible windows \code{T*q^j}
and to automatically select an optimal window for each time series.
Default is \eqn{3/4}. This argument is ignored if the \code{Window} is set by the user.}

\item{j}{numeric vector to define the set of possible windows \code{T*q^j}
and to automatically select an optimal window for each time series.
Default is \code{c(8:11)}. This argument is ignored if the \code{Window} is set by the user.}

\item{ar.order}{order of the autoregressive filter when \code{BIC = FALSE},
or the maximal order for BIC-based filtering. Default is \code{round(10*log10(T))}.
The \code{ar.order} can be a scalar or \eqn{N}-vector. If scalar, the same
\code{ar.order} is applied to each time series. An \eqn{N}-vector specifies
a separate \code{ar.order} for each time series.}

\item{ar.method}{method of estimating autoregression coefficients.
Default \code{"HVK"} delivers robust difference-based estimates by
\insertCite{Hall_VanKeilegom_2003;textual}{funtimes}. Alternatively,
options of \command{ar} function can be used, such as \code{"burg"},
\code{"ols"}, \code{"mle"}, and \code{"yw"}.}

\item{BIC}{logical value indicates whether the order of autoregressive
filter should be selected by Bayesian information criterion (BIC).
If \code{TRUE} (default), models of orders \eqn{p=} 0,1,...,\code{ar.order}
or \eqn{p=} 0,1,...,\code{round(10*log10(length(x)))} are considered,
depending on whether \code{ar.order} is defined or not
(\code{x} is the time series).}
}
\value{
A list of class \code{"htest"} containing the following components:
\item{method}{name of the method.}
\item{data.name}{name of the data.}
\item{statistic}{value of the test statistic.}
\item{p.value}{\eqn{p}-value of the test.}
\item{alternative}{alternative hypothesis.}
\item{estimate}{list with elements \code{common_trend_estimates},
\code{ar_order_used}, \code{Window_used}, \code{wavk_obs}, and
\code{all_considered_windows}. The latter is a table with bootstrap and
asymptotic test results for all considered windows, that is, without adaptive
selection of the local window.}
}
\description{
Nonparametric test for synchronicity of parametric trends in multiple time series
\insertCite{Lyubchich_Gel_2016_synchronism}{funtimes}.
The method tests whether \eqn{N} observed time series exhibit the same trend
of some pre-specified smooth parametric form.
}
\details{
Arguments \code{Window}, \code{j}, and \code{q} are used to set windows
for the local regression. Current version of the function assumes two options:
(1) user specifies one fixed window for each time series using the argument
\code{Window} (if \code{Window} is set, \code{j} and \code{q} are ignored),
and (2) user specifies a set of windows by \code{j} and \code{q} to apply
this set to each time series and to select an optimal window using a heuristic
\eqn{m}-out-of-\eqn{n} subsampling algorithm \insertCite{Bickel_Sakov_2008}{funtimes}.
The option of selecting windows automatically for some of the time series,
while for other time series the window is fixed, is not available yet.
If none of these three arguments is set, default \code{j} and \code{q} are used.
Values \code{T*q^j} are mapped to the largest previous integer, then only
those greater than 2 are used.

See more details in \insertCite{Lyubchich_Gel_2016_synchronism;textual}{funtimes}
and \insertCite{Lyubchich_2016_trends;textual}{funtimes}.
}
\examples{
#Fix seed for reproducible simulations:
set.seed(1)

# Simulate two autoregressive time series of length n without trend 
#(i.e., with zero or constant trend) 
# and arrange the series into a matrix:
n <- 200
y1 <- arima.sim(n = n, list(order = c(1, 0, 0), ar = c(0.6)))
y2 <- arima.sim(n = n, list(order = c(1, 0, 0), ar = c(-0.2)))
Y <- cbind(y1, y2)
plot.ts(Y)


#Test H0 of a common linear trend:
\dontrun{
    sync_test(Y ~ t, B = 500)
}
# Sample output:
##	Nonparametric test for synchronism of parametric trends
##
##data:  Y 
##Test statistic = -0.0028999, p-value = 0.7
##alternative hypothesis: common trend is not of the form Y ~ t.
##sample estimates:
##$common_trend_estimates
##               Estimate Std. Error    t value  Pr(>|t|)
##(Intercept) -0.02472566  0.1014069 -0.2438261 0.8076179
##t            0.04920529  0.1749859  0.2811958 0.7788539
##
##$ar.order_used
##         y1 y2
##ar.order  1  1
##
##$Window_used
##       y1 y2
##Window 15  8
##
##$all_considered_windows
## Window    Statistic p-value Asympt. p-value
##      8 -0.000384583   0.728       0.9967082
##     11 -0.024994408   0.860       0.7886005
##     15 -0.047030164   0.976       0.6138976
##     20 -0.015078579   0.668       0.8714980
##
##$wavk_obs
##[1]  0.05827148 -0.06117136

# Add a time series y3 with a different linear trend and re-apply the test:
y3 <- 1 + 3*((1:n)/n) + arima.sim(n = n, list(order = c(1, 0, 0), ar = c(-0.2)))
Y2 <- cbind(Y, y3)
plot.ts(Y2)
\dontrun{
    sync_test(Y2 ~ t, B = 500)}
# Sample output:
##	Nonparametric test for synchronism of parametric trends
##
##data:  Y2 
##Test statistic = 0.48579, p-value < 2.2e-16
##alternative hypothesis: common trend is not of the form Y2 ~ t.
##sample estimates:
##$common_trend_estimates
##              Estimate Std. Error  t value     Pr(>|t|)
##(Intercept) -0.3632963 0.07932649 -4.57976 8.219360e-06
##t            0.7229777 0.13688429  5.28167 3.356552e-07
##
##$ar.order_used
##         Y.y1 Y.y2 y3
##ar.order    1    1  0
##
##$Window_used
##       Y.y1 Y.y2 y3
##Window    8   11  8
##
##$all_considered_windows
## Window Statistic p-value Asympt. p-value
##      8 0.4930069       0    1.207378e-05
##     11 0.5637067       0    5.620248e-07
##     15 0.6369703       0    1.566057e-08
##     20 0.7431621       0    4.201484e-11
##
##$wavk_obs
##[1]  0.08941797 -0.07985614  0.34672734

#Other hypothesized trend forms can be specified, for example:
\dontrun{
    sync_test(Y ~ 1) #constant trend
    sync_test(Y ~ poly(t, 2)) #quadratic trend
    sync_test(Y ~ poly(t, 3)) #cubic trend
}

}
\references{
\insertAllCited{}
}
\seealso{
\code{\link[stats]{ar}}, \code{\link{HVK}}, \code{\link{WAVK}},
\code{\link{wavk_test}}
}
\author{
Yulia R. Gel, Vyacheslav Lyubchich, Ethan Schaeffer, Xingyu Wang
}
\keyword{htest}
\keyword{synchrony}
\keyword{trend}
\keyword{ts}
