\name{frscore}
\alias{frscore}
\title{frscore}
\usage{
frscore(
  sols,
  dat = NULL,
  normalize = c("truemax", "idealmax", "none"),
  maxsols = 50,
  verbose = FALSE,
  print.all = FALSE,
  comp.method = c("causal_submodel", "is.submodel")
)
}
\arguments{
\item{sols}{Character vector of class "stdAtomic" or "stdComplex" (as generated by \code{\link[cna:cna]{cna()}}) that contains the solutions/models to be scored.}

\item{dat}{A \code{configTable}, a data frame, a matrix, or a list that specifies the range of admissible factor values
for the factors featured in the models included in \code{sols}. Only needed when the models in
\code{sols} are multi-valued, otherwise ignored.}

\item{normalize}{String that determines the method used in
normalizing the scores. \code{"truemax"} (default) normalizes by the highest score among the elements of \code{sols},
such that the highest scoring solution types get score 1. \code{"idealmax"}
normalizes by a theoretical maximum score (see Details). }

\item{maxsols}{Integer determining the maximum number of unique solution
types found in \code{sols} to be included in the scoring (see Details).}

\item{verbose}{Logical; if \code{TRUE}, additional
information about causal compatibility relations among the unique solution types found
in \code{sols} is printed. Defaults to \code{FALSE}.}

\item{print.all}{Logical, controls the number of entries printed when
printing the results. If \code{TRUE}, results are printed as when using the defaults of
\code{print.data.frame}. If \code{FALSE}, 20 highest scoring
solutions/models are printed.}

\item{comp.method}{String that determines how the models in \code{sols} are compared
to determine their fr-score. \code{"causal_submodel"} (the default) checks for causal submodel relations using \code{causal_submodel()},
\code{"is.submodel"} checks for syntactic submodel relations with \code{is.submodel()}}

}
\value{
A named list where the first element is a data frame containing
the unique solution/model types and their scores. Rest of the elements
contain additional information about the submodel relations among
the unique solutions types and about how
the function was called.
}
\description{
Calculate fit-robustness scores for a set of \code{cna}
solutions/models
}
\details{
\code{frscore()} implements fit-robustness scoring as introduced in Parkkinen and Baumgartner (2021). The function calculates the
fit-robustness scores of Boolean solutions/models output by the \code{\link[cna:cna]{cna()}} function of the \pkg{cna} package. The solutions are given to \code{frscore()} as a character vector \code{sols} obtained by reanalyzing
a data set repeatedly, e.g. with \code{rean_cna()}, using different consistency and coverage
thresholds in each analysis.

For multi-valued
models, the range of admissible values for the factors featured
in the models must be provided via the argument \code{dat}, which accepts a data
frame, \code{configTable}, or a list of factor-value ranges as its value,
in the same manner as \code{\link[cna:full.ct]{cna::full.ct()}}.
Typically, one would use the data set that the models in \code{sols}
were inferred from, and this is what is done automatically when
\code{frscore()} is called within \code{frscored_cna()}. When the models in \code{sols}
are binary, \code{dat} should be left to its default value \code{NULL},
and will in any case be ignored.

The fit-robustness scores can be normalized in two ways. In the default setting \code{normalize = "truemax"}, the score of each \code{sols[i]} is divided by the maximum score obtained by an element of \code{sols}. In case of \code{normalize = "idealmax"}, the score is normalized not by an actually obtained
maximum but by an idealized maximum, which is calculated by assuming that all solutions of equal
complexity in \code{sols} are identical and that for every \code{sols[i]} of a given complexity, all less complex
elements of \code{sols} are its submodels and all more complex elements of \code{sols} are its supermodels.
When normalization is applied, the normalized score is shown in its own column \code{norm.score} in
the results. The raw scores are shown in the column \code{score}.

If the size of the consistency and coverage interval scanned in the reanalysis series generating \code{sols} is large or there are many model ambiguities, \code{sols} may contain so many different types of solutions/models that robustness cannot be calculated for all of them in reasonable time. In that case, the argument \code{maxsols} allows for capping the number of solution types to be included in the scoring (defaults to 50). \code{frscore()} then selects the most frequent solutions/models in \code{sols} of each complexity level until  \code{maxsols} is reached and only scores the thus selected elements of \code{sols}.

If the argument \code{verbose} is set to \code{TRUE}, \code{frscore()} also prints a list indicating for each \code{sols[i]} how many raw score points it receives from which elements of \code{sols}. The verbose list is ordered with decreasing fit robustness scores.

}
\references{
V.P. Parkkinen and M. Baumgartner (2021), \dQuote{Robustness and Model Selection in Configurational Causal Modeling,} \emph{Sociological Methods and Research}, doi:10.1177/0049124120986200.

Basurto, Xavier. 2013. \dQuote{Linking Multi-Level Governance to Local Common-Pool
Resource Theory using Fuzzy-Set Qualitative Comparative Analysis: Insights from
Twenty Years of Biodiversity Conservation in Costa Rica.} \emph{Global Environmental Change} \strong{23} (3):573-87.

}
\seealso{\code{\link[frscore:rean_cna]{rean_cna()}}, \code{\link[cna:cna]{cna()}},
\code{\link[frscore:causal_submodel]{causal_submodel()}}, \code{\link[cna:is.submodel]{is.submodel()}}}
\examples{
# Artificial data from Parkkinen and Baumgartner (2021)
sols1 <- rean_cna(d.error, attempt = seq(1, 0.8, -0.1))
sols1 <- do.call(rbind, sols1)
frscore(sols1$condition)

\donttest{
# Real fuzzy-set data from Basurto (2013)
sols2 <- rean_cna(d.autonomy, type="fs", ordering = list("EM", "SP"),
         strict = TRUE, maxstep = c(3,3,9))
sols2 <- do.call(rbind, sols2)$condition  # there are 217 solutions
# At the default maxsols only 50 of those solutions are scored.
frscore(sols2)
# By increasing maxsols the number of solutions to be scored can be controlled.
frscore(sols2, maxsols = 100)


# Multi-valued data/models (data from Hartmann and Kemmerzell (2010))
# Short reanalysis series, change `attempt` value to mimick a more realistic use case
sols3 <- rean_cna(d.pban, outcome = "PB=1", attempt = seq(0.8, 0.7, -0.1), type = "mv")
sols3 <- do.call(rbind, sols3)$condition
# For mv data, frscore() needs the data to determine admissible factor values
frscore(sols3, dat = d.pban)

# Changing the normalization
frscore(sols2, normalize = "none")
frscore(sols2, normalize = "truemax")
frscore(sols2, normalize = "idealmax")

# verbose
frscore(sols2, maxsols = 20, verbose = TRUE)
}



}
