% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lm_table.R
\name{lm_table}
\alias{lm_table}
\title{Fit linear regressions by group, and get different output options.}
\usage{
lm_table(df, model, .groups = NA, output = "table", est.name = "est",
  keep_model = FALSE)
}
\arguments{
\item{df}{A data frame.}

\item{model}{A linear regression model, with or without quotes. The variables mentioned in the model must exist in the provided data frame. X and Y sides of the model must be separated by "~".}

\item{.groups}{Optional argument. Quoted name(s) of grouping variables used to fit multiple regressions, one for each level of the provided variable(s). Default \code{NA}.}

\item{output}{Selects different output options. Can be either \code{"table"}, \code{"merge"}, \code{"merge_est"} and \code{"nest"}. See details for explanations for each option. Default: \code{"table"}.}

\item{est.name}{Name of the estimated y value. Used only if \code{est.name = TRUE}. Default: \code{"est"}.}

\item{keep_model}{If \code{TRUE}, a column containing lm object(s) is kept in the output. Useful if the user desires to get more information on the regression.Default: \code{FALSE}.}
}
\value{
A data frame. Different data frame options are available using the output argument.
}
\description{
With this function it's possible to fit linear regressions by a grouping variable, and get a data frame
with each column as a coefficient and quality of fit variables, and other output options. Works with dplyr grouping functions.
}
\details{
With this function there's no more need to use the \code{do} function when fitting a linear regression in a pipe line.
It's also possible to easily make fit multiple regressions, specifying a grouping variable.
In addition to that, the default output sets each coefficient as a column, making it easy to call coefficients by name or position
when estimating values. 

It's possible to use the \code{output} argument to get a merged table if \code{output="merge"}, that binds
the original data frame and the fitted coefficients. 
If \code{output="merge_est"} we get a merged table as well, but with y estimated using the coefficients. If the fit is made using groups, this is taken into account, i.e. the estimation is made by group.

If \code{output="nest"}, a data frame with nested columns is provided. This can be used if the user desires to get a customized output.
}
\examples{
library(forestmangr)
library(dplyr)

data("exfm19")
exfm19

# Fit Schumacher and Hall model for volume estimation, and get
# coefficient, R2 and error values:

lm_table(exfm19, log(VWB) ~  log(DBH) + log(TH))   

# Fit SH model by group:
lm_table(exfm19, log(VWB) ~  log(DBH) + log(TH), "STRATA")

# This can also be done using dplyr::group_by:
exfm19 \%>\% 
  group_by(STRATA) \%>\% 
  lm_table(log(VWB) ~  log(DBH) + log(TH) )
  
# It's possible to merge the original data with the table containg the coefficients
# using the output parameter:
fit <- lm_table(exfm19, log(VWB) ~  log(DBH) + log(TH), "STRATA", output = "merge")
head(fit)

# It's possible to merge the original data with the table,
# and get the estimated values for this model:
fit <- lm_table(exfm19, log(VWB) ~  log(DBH) + log(TH),"STRATA",
 output = "merge_est", est.name = "VWB_EST") 
head(fit)
       
# It's possible to further customize the output,
# unnesting the nested variables provided when output is defined as "nest":
lm_table(exfm19, log(VWB) ~  log(DBH) + log(TH),"STRATA", output = "nest")

}
\author{
Sollano Rabelo Braga \email{sollanorb@gmail.com}
}
