#' Plot Mann–Kendall–Sneyers (MKS) Test Results
#'
#' Constructs a two‐panel visualization of the MKS test. The upper panel plots the 
#' normalized progressive and regressive Mann–Kendall S‐statistics over time, with 
#' dashed confidence bounds and potential trend‐change points. The lower panel 
#' contains the annual maximum series data with the change points highlighted.
#'
#' @param results A list generated by [eda_mks_test()].
#'
#' @param show_line If `TRUE` (default), draw a fitted line through the data.
#'
#' @param ... Optional named arguments: 'title', 'top_xlabel', 'top_ylabel', 
#' 'bottom_xlabel' and 'bottom_ylabel'.
#'
#' @return A 'patchwork' object with two 'ggplot2' panels stacked vertically.
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' years <- seq(from = 1901, to = 2000)
#' results <- eda_mks_test(data, years)
#' plot_mks_test(results)
#'
#' @seealso [eda_mks_test()]
#'
#' @import ggplot2
#' @import patchwork
#' @export

plot_mks_test <- function(results, show_line = TRUE, ...) {

	# Create a dataframe for the bounds
	bound_df <- data.frame(y = c(-results$bound, results$bound))

	# Create the plotting dataframe
	df <- data.frame(
		max = results$data,
		year = results$years,
		progressive = results$progressive_series,
		regressive = results$regressive_series
	)

	# Capture optional arguments
	args <- list(...)

	# Define labels for the plot 
	title <- args$title %||% "Mann-Kendall-Sneyers Test"
	top_xlabel <- args$top_xlabel %||% "Year" 
	top_ylabel <- args$top_ylabel %||% "Normalized Trend Statistic"
	bottom_xlabel <- args$bottom_xlabel %||% "Year" 
	bottom_ylabel <- args$bottom_ylabel %||% expression(Streamflow ~ m^3/s)

	# Plot the normalized trend statistics and confidence bands
	p1 <- ggplot(df, aes(x = .data$year)) +
		geom_line(aes(color = "black", y = .data$progressive), linewidth = 1.2) +
		geom_line(aes(color = "gray",  y = .data$regressive), linewidth = 1.2) +
		geom_hline(
			data = bound_df,
			aes(yintercept = .data$y, color = "red"),
			linewidth = 1.2,
			linetype = "dashed",
		) +
		geom_point(
			data = results$change_points,
			aes(x =.data$year, y = .data$statistic, color = "blue"), 
			size = 4
		) +
		labs(title = title, x = top_xlabel, y = top_ylabel, color = "Legend") +
		scale_color_manual(
			values = c("black" = "black","gray" = "gray","blue" = "blue","red" = "red"),
			breaks = c("black", "gray", "red", "blue"),
			labels = c(
				"Progressive Series",
				"Regressive Series",
				"Confidence Bounds", 
				"Potential Trend Change"
			)
		)

	# Plot the change points on the original dataset
	p2 <- ggplot(df, aes(x = .data$year, y = .data$max)) +
		geom_point(aes(color = "dimgray"), size = 2.25) +
		(if (show_line) geom_line(color = "dimgray", linewidth = 1) else NULL) +
		geom_point(
			data = results$change_points,
			aes(x = .data$year, y = .data$value, color = "blue"), 
			size = 4
		) +
		labs(x = bottom_xlabel, y = bottom_ylabel, color = "Legend") +
		scale_color_manual(
			values = c("dimgray" = "dimgray", "blue" = "blue"),
			breaks = c("dimgray", "blue"),
			labels = c(bottom_ylabel, "Potential Trend Change")
		)	

	# Stack plots on top of each other and return
	(add_theme(add_scales(p1)) / add_theme(add_scales(p2))) + plot_annotation(title = "")

	
}
