% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FacebookReactionsCollection.R
\name{FacebookReactionsCollection}
\alias{FacebookReactionsCollection}
\title{Build a collection of Facebook reactions to posts and comments}
\usage{
FacebookReactionsCollection(id, token = NULL, parameters = list(),
  fields = c("id", "name", "type", "profile_type"),
  n = getOption("facebook.maxitems"), metadata = FALSE,
  .progress = create_progress_bar())
}
\arguments{
\item{id}{A character vector or a comma-delimited string of IDs or an existing Facebook Collection of any 
of the supported types (see below).}

\item{token}{Either a temporary access token created at
\url{https://developers.facebook.com/tools/explorer} or the OAuth token 
created with \code{\link{fbOAuth}}. If token \code{NULL} and \code{id} is a collection,
use the token of the source collection. Otherwise, no query is performed
to the Facebook Graph API and an empty collection is returned.}

\item{parameters}{A list of parameters to be added to the Facebook Graph API query. For more information on the
accepted parameters, see: \url{https://developers.facebook.com/docs/graph-api/using-graph-api}.}

\item{fields}{A character vector with the fields to get for each \code{id}. If no value
for a given field is found, it will be set to \code{NULL}.}

\item{n}{If \code{id} is an iterable collection, then \code{n} is the maximum number of reaction to be pulled for each element of the source collection
in \code{id}. It can be set to \code{Inf} to pull out any available public reaction and assumes the default value from the value
of \code{facebook.maxitems} global option if missing. If \code{id} is not a collection or cannot be iterated, the parameter is ignored.}

\item{metadata}{If set to \code{TRUE}, the metadata for each ID is pulled with the data
and the \code{type} slot is fed accordingly.
Please note that setting this to \code{TRUE} could considerably
slow down the execution time, as more queries are needed.}

\item{.progress}{progress_bar object as defined in the plyr package.
By default the \code{none} progress bar is used, which prints nothing to the console. See \link[plyr]{create_progress_bar} for details.}
}
\value{
A collection of reactions in a \code{\link{FacebookReactionsCollection-class}} object.
}
\description{
Connect to Facebook Graph API, get information from a list of reactions to public posts or comments and build a \code{FacebookReactionsCollection-class}
instance.
}
\details{
\code{FacebookReactionsCollection} is the constructor for the \code{\link{FacebookReactionsCollection-class}}.
It returns data about reactions to posts or comments but doesn't return the comments or posts themselves.

Since Facebook doesn't provide a key for a single reaction, the ID slot for this kind of collection doesn't uniquely identify a reaction on Facebook.
The \code{id} (the user who expressed the reaction) coupled with the \code{parent} (the place where she put the like) identifies a unique key for the reaction

As a consequence, you cannot build a reactions collection starting from atomic IDs, but you must pass an instance of a Facebook Posts Collection or a Facebook Comments Collection built using the construction \code{\link{FacebookPostsCollection}}
or \code{\link{FacebookCommentsCollection}} as \code{id} parameter.
}
\section{Nesting fields}{

Due to the network-graph nature of Facebook data model,
you can specify fields details for each field nesting \code{.fields()} clauses.

For example, if you need only \code{id} and \code{source} for the \code{cover} field, this is valid among others:
\code{cover.fields(id,source)}.

Following the same philosophy, if you need only \code{id} and \code{name} for the \code{from} node
you can use \code{from.fields(id,name)}.
}

\section{Valid sources}{

Instead of a character vector, one of these collections can also be passed as parameter in \code{id}:
\itemize{
 \item{\code{\link{FacebookPostsCollection-class}} will build a collection with 
 all the likes to the posts of the source collection}
 \item{\code{\link{FacebookCommentsCollection-class}} will build a collection with 
 all the likes to the comments of the source collection}
 \item{\code{\link{FacebookUsersCollection-class}} will build a collection with 
 the posts written on the walls of the users in the source collection.}
 \item{\code{\link{FacebookPhotosCollection-class}} will build a collection with 
 the likes to the photos in the source collection. The author of the photo may grant the
 \code{user_photos} to the current application to perform this action.}
 \item{\code{\link{FacebookVideosCollection-class}} will build a collection with 
 the likes to the videos in the source collection. The author of the photo may grant the
 \code{user_videos} to the current application to perform this action.}
}
}

\examples{
\dontrun{
## See examples for fbOAuth to know how token was created.
 load("fb_oauth")
 
## Getting information about two example Facebook Pages
 fb.pages <- FacebookPagesCollection(id = c("9thcirclegames",
                                            "NathanNeverSergioBonelliEditore"),
                                     token = fb_oauth)
 
## Pull the latest 10 posts from each page in a pages collection
 fb.posts <- FacebookPostscollection(id = fb.pages, token = fb_oauth, n = 10)
 
## Pull all the reactions from each element of the posts collection
fb.posts.reactions <- FacebookReactionssCollection(fb.posts, fb_token, n = Inf)
 
## Pull all the available comments from each post of the post collection
 fb.comments <- FacebookPostscollection(id = fb.posts, token = fb_oauth, n = Inf)
 
## Pull all the likes from each element of the comments collections
 fb.comments.reactions <- FacebookReactionsCollection(id = fb.comments, token = fb_oauth, n = Inf)  
   
## Convert the collection to a data frame
fb.posts.reactions.df <- as.data.frame(fb.posts.reactions)

# The same as before in a more compact fashion using the pipe operator
# chaining from a Pages then to a Posts Collection and finally building a Likes Collection
fb.posts.likes.pipe <- 
 FacebookPagesCollection(id = c("9thcirclegames",
                                "NathanNeverSergioBonelliEditore"),
                         token = fb_oauth) \%>\%
     FacebookPostscollection(n = 10) \%>\%
     FacebookReactionsCollection(n = Inf)
}

}
\seealso{
\code{\link{FacebookCommentsCollection}},
\code{\link{FacebookPostsCollection}},
#' \code{\link{FacebookLikesCollection}},
\code{\link{facebook.object.likes}},
\code{\link{fbOAuth}}

Other Facebook Collection Constructors: \code{\link{FacebookAlbumsCollection}},
  \code{\link{FacebookCommentsCollection}},
  \code{\link{FacebookConversationsCollection}},
  \code{\link{FacebookEventsCollection}},
  \code{\link{FacebookGroupsCollection}},
  \code{\link{FacebookLikesCollection}},
  \code{\link{FacebookMessagesCollection}},
  \code{\link{FacebookPagesCollection}},
  \code{\link{FacebookPhotosCollection}},
  \code{\link{FacebookPostsCollection}},
  \code{\link{FacebookUsersCollection}},
  \code{\link{FacebookVideosCollection}}
}
\author{
Gabriele Baldassarre \url{https://gabrielebaldassarre.com}
}
