% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spatial_parabola.R
\name{spatial_parabola}
\alias{spatial_parabola}
\title{Locate signals of a seismic event by time difference parabola overlay}
\usage{
spatial_parabola(data, d_map, v, dt, plot, ...)
}
\arguments{
\item{data}{\code{Numeric} matrix or \code{eseis} object, seismic signals 
to cross-correlate.}

\item{d_map}{\code{List} object, distance maps for each station. Output 
of \code{spatial_distance}.}

\item{v}{\code{Numeric} value or vector, apparent seismic wave velocity
(m/s).}

\item{dt}{\code{Numeric} value, sampling period.}

\item{plot}{\code{Character} value, keyword defining if or which output 
is to be plotted. If omitted, no plot output is generated. If set to 
\code{"parabola"}, a plot with all overlaid station pair parabolas is 
created. If set to \code{"offset"}, a plot of the average time offset 
of each grid cell is created.}

\item{\dots}{Additional arguments passed to the plot function.}
}
\value{
A terra raster with average travel time offsets for each grid 
cell, implying the most likely source location coinciding with the 
smallest offset value.
}
\description{
The function performs event location in space by finding the grid cell 
with minimum average travel time difference using the parabola approach. For 
further information see also Hibert et al. (2014) DOI: 10.1002/2013JF002970.
}
\examples{

\dontrun{

## create synthetic DEM
dem <- terra::rast(nrows = 20, ncols = 20, 
                   xmin = 0, xmax = 10000, 
                   ymin= 0, ymax = 10000, 
                   vals = rep(0, 400))

## define station coordinates
sta <- data.frame(x = c(1000, 9000, 5000),
                  y = c(1000, 1000, 9000),
                  ID = c("A", "B", "C"))

## create synthetic signal (source in the center of the DEM)
s <- rbind(dnorm(x = 1:1000, mean = 400, sd = 50),
           dnorm(x = 1:1000, mean = 400, sd = 50),
           dnorm(x = 1:1000, mean = 800, sd = 50))

## plot DEM and stations
terra::plot(dem)

text(x = sta$x, 
     y = sta$y, 
     labels = sta$ID)

## calculate spatial distance maps and inter-station distances
D <- spatial_distance(stations = sta[,1:2],
                             dem = dem)
                             
## restore SpatRaster object for plotting purpose
D_map_1 <- terra::rast(crs = D$maps[[1]]$crs,
                       ext = D$maps[[1]]$ext,
                       res = D$maps[[1]]$res,
                       val = D$maps[[1]]$val)
                      
## plot distance map
terra::plot(D_map_1) 

## locate signal
e <- spatial_parabola(data = s, 
                      d_map = D$maps, 
                      v = 1000, 
                      dt = 1/100,
                      plot = "parabola",
                      zlim = c(0, 2))
}

}
\author{
Michael Dietze, Clement Hibert (ITES Strasbourg)
}
