% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/episcan.R
\name{episcan}
\alias{episcan}
\title{Scan pairwise epistasis}
\usage{
episcan(geno1, geno2 = NULL, pheno = NULL,
  phetype = c("case-control", "quantitative"), outfile = "episcan",
  suffix = ".txt", zpthres = 1e-06, chunksize = 1000, scale = TRUE,
  ...)
}
\arguments{
\item{geno1}{a data.frame or matrix of the first genotype data. \code{big.matrix} object from \pkg{bigmemory} also works.
The columns contain the information of variables and the rows contain the information of samples.}

\item{geno2}{optional. A data.frame or matrix of the second genotype data. \code{big.matrix} object from \pkg{bigmemory} also works.
The columns contain the information of variables and the rows contain the information of samples.}

\item{pheno}{a vector (named or not). If not provided, the value of \code{geno2} will be used if it is a vector. The values is either case-control phenotype (0, 1) or quantitative phenotype.}

\item{phetype}{character string. Either "case-control" or "quantitative".}

\item{outfile}{output file name. Default is "episcan".}

\item{suffix}{suffix for output file. Default is ".txt". The final result will be stored in \code{outfile}\code{suffix}.}

\item{zpthres}{is the significance threshold to select variant pairs for output. Default is 1e-6.}

\item{chunksize}{the number of variants in each chunk.}

\item{scale}{a logical value to define wheter the input data needs to be normalized. Default is TRUE which means, by default, 
all the genotype data will be normalized and if the phetype is "quantitative", the phenotype will also be normalized.}

\item{...}{not used.}
}
\value{
null
}
\description{
Genomic interaction analysis with EPIBLASTER or epistasis-oriented Hilbert–Schmidt Independence Criterion (HSIC).
}
\examples{
# simulate some data
set.seed(123)
geno1 <- matrix(sample(0:2, size = 1000, replace = TRUE, prob = c(0.5, 0.3, 0.2)), 
ncol = 10)
geno2 <- matrix(sample(0:2, size = 2000, replace = TRUE, prob = c(0.4, 0.3, 0.3)), 
ncol = 20)
dimnames(geno1) <- list(row = paste0("IND", 1:nrow(geno1)), 
col = paste0("rs", 1:ncol(geno1)))
dimnames(geno2) <- list(row = paste0("IND", 1:nrow(geno2)), 
col = paste0("exm", 1:ncol(geno2)))
p1 <- c(rep(0, 60), rep(1, 40))
p2 <- rnorm(100)

# one genotype with case-control phenotype
episcan(geno1 = geno1, 
geno2 = NULL, 
pheno = p1, 
phetype = "case-control",
outfile = "episcan_1geno_cc", 
suffix = ".txt", 
zpthres = 0.9, 
chunksize = 10, 
scale = TRUE)

# take a look at the result
res <- read.table("episcan_1geno_cc.txt", 
header = TRUE, 
stringsAsFactors = FALSE)
head(res)

# two genotypes with quantitative phenotype
episcan(geno1 = geno1, 
geno2 = geno2, 
pheno = p2, 
phetype = "quantitative",
outfile = "episcan_2geno_quant", 
suffix = ".txt", 
zpthres = 0.9, 
chunksize = 10, 
scale = TRUE)
}
\references{
Kam-Thong, T., D. Czamara, K. Tsuda, K. Borgwardt, C. M. Lewis, A. Erhardt-Lehmann, B. Hemmer, et al. 2011. 
"EPIBLASTER-Fast Exhaustive Two-Locus Epistasis Detection Strategy Using Graphical Processing Units." Journal Article. 
European Journal of Human Genetics 19 (4): 465–71. https://doi.org/10.1038/ejhg.2010.196. 

Kam-Thong, T., B. Pütz, N. Karbalai, B. Müller-Myhsok, and K. Borgwardt. 2011. "Epistasis Detection on Quantitative 
Phenotypes by Exhaustive Enumeration Using GPUs." Journal Article. Bioinformatics 27 (13): i214–21. https://doi.org/10.1093/bioinformatics/btr218.
}
\author{
Beibei Jiang \email{beibei_jiang@psych.mpg.de}
}
