% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_obj_address.r
\name{get_obj_address}
\alias{get_obj_address}
\title{Return the memory address of an object}
\usage{
get_obj_address(obj, envir = NULL, envmap = NULL, n = 0,
  include_functions = FALSE)
}
\arguments{
\item{obj}{object whose memory address is requested. It can be given as a variable name or an expression.
Strings representing object names are not interpreted and return \code{NULL}.}

\item{envir}{environment where the object should be searched for. All parent environments of
\code{envir} are searched as well. It defaults to \code{NULL} which means that it should be searched in the
whole workspace (including packages, namespaces, and user-defined environments).}

\item{envmap}{data frame containing a lookup table with name-address pairs of environment names and
addresses to be used when searching for environment \code{env}. It defaults to \code{NULL} which means that the
lookup table is constructed on the fly with the environments defined in the \code{envir} environment
--if not \code{NULL}--, or in the whole workspace if \code{envir=sNULL}.
See the details section for more information on its structure.}

\item{n}{number of levels to go up from the calling function environment to resolve the name
of \code{obj}. It defaults to 0 which implies the calling environment.}

\item{include_functions}{whether to include funtion execution environments as environments where the object
is searched for. Set this flag to \code{TRUE} with caution because there may be several functions where the
same object is defined, for instance functions that are called as part of the object searching process!}
}
\value{
The 8-digit (32-bit architectures) thru 16-digit (64-bit architectures) memory address of
the input object given as a string enclosed in <>  (e.g. \code{"<0000000005E90988>"})
(note that Ubuntu Debian may use 12-digit memory addresses),
or \code{NULL} under any of the following situations:
\itemize{
\item the object is \code{NULL}, \code{NA}, or a string, or any other object whose memory address changes every
time the object is referred to (for instance for \code{alist[1]} --as opposed to \code{alist[[1]]}--
where \code{alist} is a list.
\item the object is a constant (e.g. \code{TRUE}, \code{3}, etc.)
\item the object does not exist in the given environment.
\item the object is an expression that cannot be evaluated in the given environment.
}

Note that for the last case, although constants have a memory address, this address is meaningless as
it changes with every invocation of the function. For instance, running
\code{address(3)} several times will show a different memory address each time, and that is why
\code{get_obj_address} returns \code{NULL} in those cases.

When \code{envir=NULL} (the default) or when an object exists in several environments,
the memory address is returned for all of the environments where the object is found. In that case, the addresses are
stored in an array whose names attribute shows the environments where the object is found.
}
\description{
Return the memory address of an object after recursively searching for the object in all the environments defined
in a specified environment or in all the environments defined in the whole workspace.
}
\details{
The object is first searched recursively in all environments defined in the specified environment (if any),
by calling \code{obj_find}.
If no environment is specified, the object is searched recursively in the whole workspace.

The memory address is then retrieved for every object found in those environments having the same name
as the given object \code{obj}.

Strings return \code{NULL} but strings can be the result of an expression passed as argument to this function.
In that case, the string is interpreted as an object and its memory address is returned as long as
the object exists.

If \code{envmap} is passed it should be a data frame providing an address-name pair lookup table
of environments and should contain at least the following columns:
\itemize{
\item{\code{location}} for user-defined environments, the name of the environment where the environment
is located; otherwise \code{NA}.
\item{\code{pathname}} the full \emph{environment path} to reach the environment separated by \code{$}
(e.g. \code{"env1$env$envx"})
\item{\code{address}} an 8-digit (32-bit architectures) thru 16-digit (64-bit architectures) memory address
of the environment given in \code{pathname} enclosed in < > (e.g. \code{"<0000000007DCFB38>"}
(64-bit architectures))
Be ware that Linux Debian distributions may have a 12-digit memory address representation.
So the best way to know is to check a memory address by calling e.g. `address("x")`.
}
Passing an \code{envmap} lookup table is useful for speedup purposes, in case several calls to this
function will be performed in the context of an unchanged set of defined environments.
Such \code{envmap} data frame can be created by calling \link{get_env_names}.
Use this parameter with care, as the matrix passed may not correspond to the actual mapping of existing
environments to their addresses and in that case results may be different from those expected.
}
\examples{
env1 = new.env()
env1$x = 3                       # x defined in environment 'env1'
x = 4                            # x defined in the Global Environment
get_obj_address(env1$x)          # returns the memory address of the object 'x'
                                 # defined in the 'env1' environment
get_obj_address(x, envir=env1)   # same as above
get_obj_address(x)               # Searches for object 'x' everywhere in the workspace and
                                 # returns a named array with the memory address of all its
                                 # occurrences, where the names are the names of the
                                 # environments where x was found.

# Memory addresses of objects whose names are stored in an array and retrieved using sapply()
env1$y <- 2;
objects <- c("x", "y")
sapply(objects, FUN=get_obj_address, envir=env1)	# Note that the address of object "x"
                                                 # is the same as the one returned above
                                                 # by get_obj_address(x, envir=env1)

# Memory address of elements of a list
alist <- list("x")
get_obj_address(alist[[1]])      # memory address of object 'x'
get_obj_address(alist[1])        # NULL because alist[1] has a memory address
                                 # that changes every time alist[1] is referred to.
}
