#' MovingIncubation simulate incubation of a nest with the beginning varying day by day
#' @title Simulate incubation of a nest with the beginning of incubation varying
#' @author Marc Girondot
#' @return A dataframe with informations about thermosensitive period length and incubation length day by day of incubation
#' @param NestsResult A result file generated by searchR
#' @param resultmcmc A mcmc result. Will be used rather than SE if provided.
#' @param SexualisationTRN A model for sexualisation thermal reaction norm during TSP obtained using STRN()
#' @param temperatures.df A data.frame with 2 or 3 columns: Times, Temperatures and Temperatures.end.incubation (facultative)
#' @param temperature.heterogeneity SD of heterogeneity of temperatures. Can be 2 values, sd_low and sd_high and then HelpersMG::r2norm() is used.
#' @param average.incubation.duration The average time to complete incubation (not used if metabolic heating is setup)
#' @param metabolic.heating Degrees Celsius to be added at the end of incubation due to metabolic heating
#' @param skip Number of data to skip between two runs
#' @param parameters A set of parameters if result is not provided.
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param SE Standard error for each parameter if not present in result is not provided
#' @param hessian A hessian matrix
#' @param GTRN.CI How to estimate CI for embryo growth thermal reaction norm; can be NULL, "SE", "MCMC", or "Hessian".
#' @param SexualisationTRN.CI How to estimate CI of sexualisation thermal reaction norm. Can be NULL, "SE", "MCMC", or "Hessian".
#' @param SexualisationTRN.mcmc MCMC object for STRN.
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings as a vector ie test=c(Mean=xx, SD=yy)
#' @param M0 Measure of hatchling size proxi at laying date
#' @param TSP.borders The limits of TSP
#' @param embryo.stages The embryo stages. At least TSP.borders stages must be provided to estimate TSP length
#' @param replicate.CI Number of randomizations to estimate CI
#' @param parallel Should parallel computing be used. TRUE or FALSE
#' @param progressbar Should a progress bar be shown ? TRUE or FALSE
#' @description Simulate incubation of a nest with the beginning varying day by day\cr
#' Temperatures must be in a data.frame with one column (Time) being the time and the second the temperatures (Temperature). A third columns can indicate the temperature at the end of incubation (Temperature.end.incubation). Do not use FormatNests() for this dataframe.
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p_SSM4p)
#' ti <- seq(from=0, to=(60*24*100), by=60)
#' temperatures <- rnorm(length(ti), 29, 5)
#' temperatures <- temperatures+ti/(60*24*100)/2
#' layout(mat=1:3)
#' parpre <- par(mar=c(4, 4, 1, 1)+0.4)
#' plot(ti/(60*24), temperatures, type="l", xlab="Days", 
#'      ylab=expression("Nest temperature in "*degree*"C"), bty="n", las=1)
#' # The sexualisation thermal reaction norm is calculated for South Pacific RMU
#' out <- MovingIncubation(NestsResult=resultNest_4p_SSM4p, 
#'      temperatures.df=data.frame(Time=ti, Temperature=temperatures),
#'      metabolic.heating = 0, 
#'      SexualisationTRN = structure(c(71.922411148397, 613.773055147801, 
#'      318.059753164125, 120.327257089974), 
#'      .Names = c("DHA", "DHH", "T12H", "Rho25")))
#' with(out, plot(Time/(60*24), Incubation.length.mean/(60*24), 
#'      xlab="Days along the season", 
#'      ylab="Incubation duration", 
#'      type="l", bty="n", las=1, ylim=c(70, 80)))
#' with(out, plot(Time/(60*24), TSP.GrowthWeighted.STRNWeighted.temperature.mean, 
#'      xlab="Days along the season", 
#'      ylab=expression("CTE for sex ratio in "*degree*"C"), 
#'       type="l", bty="n", las=1, ylim=c(30, 31)))
#' par(mar=parpre)
#' layout(mat=c(1))
#' }
#' @export


MovingIncubation <-
  function(NestsResult=NULL, 
           resultmcmc=NULL,
           temperatures.df=stop("A data.frame must be provided"),
           metabolic.heating=0, 
           GTRN.CI="Hessian",
           temperature.heterogeneity=0, 
           average.incubation.duration=60*1440,
           skip = 1, parameters=NULL, fixed.parameters=NULL, SE=NULL, 
           hessian = NULL, 
           derivate=NULL, test=NULL, M0=NULL, TSP.borders=c(21, 26), 
           embryo.stages="Caretta caretta.SCL", 
           SexualisationTRN=NULL,
           SexualisationTRN.CI="Hessian",
           SexualisationTRN.mcmc=NULL,
           replicate.CI=1, 
           parallel=TRUE, 
           progressbar=TRUE) {
    
    # NestsResult=NULL; resultmcmc=NULL;temperatures.df=NULL; 
    # metabolic.heating=0;GTRN.CI="Hessian"; temperature.heterogeneity=0;
    # average.incubation.duration=60*1440; skip = 1; parameters=NULL; 
    # fixed.parameters=NULL; SE=NULL; hessian=NULL; derivate=NULL; test=NULL; 
    # M0=NULL; TSP.borders=c(21, 26); embryo.stages="Caretta caretta.SCL"; 
    # SexualisationTRN=NULL; SexualisationTRN.CI="Hessian"; SexualisationTRN.mcmc=NULL
    # replicate.CI=1; parallel=TRUE
    
    # maintenant il n'est plus possible qu'il n'y ait pas de temperatures
    #  
    if (!is.null(GTRN.CI)) {
      GTRN.CI <- tolower(GTRN.CI)
      GTRN.CI <- match.arg(GTRN.CI, choices = c("se", "mcmc", "hessian"))
    }
    if (!is.null(SexualisationTRN.CI)) {
      SexualisationTRN.CI <- tolower(SexualisationTRN.CI)
      SexualisationTRN.CI <- match.arg(SexualisationTRN.CI, choices = c("se", "mcmc", "hessian"))
    }
    
    if (is.list(SexualisationTRN)) SexualisationTRN <- SexualisationTRN$par
    
    
    if (is.null(test)) {
      # si tous sont pareils, je reprends les memes
      # Correction d'un bug, rajout de [[1]] dans result$test["Mean"][[1]][1] 30/7/2012
      if (all(NestsResult$test["Mean"]==NestsResult$test["Mean"][[1]][1]) & all(NestsResult$test["SD"]==NestsResult$test["SD"][[1]][1])) {
        test <- c(Mean=NestsResult$test["Mean"][[1]][1], SD=NestsResult$test["SD"][[1]][1])
      } else {	
        stop("The size at hatching must be provided.")
      }
    }
    
    times <- temperatures.df$Time
    temperatures <- temperatures.df$Temperature
    if (metabolic.heating == 0) {
      if (is.null(temperatures.df$Temperature.end.incubation)) {
        temperatures.end.incubation <- temperatures
      } else {
        temperatures.end.incubation <- temperatures.df$Temperature.end.incubation
      }
    }
    
    if (is.null(derivate)) derivate <- NestsResult$derivate
    if (is.null(M0))	M0 <- NestsResult$M0
    if (is.null(fixed.parameters)) fixed.parameters <- NestsResult$fixed.parameters
    if (is.null(parameters)) parameters <- NestsResult$par
    
    if (is.null(SexualisationTRN)) SexualisationTRN <- NestsResult$SexualisationTRN
    
    # je peux indiquer des SE en plus de ceux de result
    if (is.null(hessian)) {
      hessian <- NestsResult$hessian
      if (is.null(hessian)) {
        SE <- SEfromHessian(hessian = hessian)
      }
    }
    if (is.null(SE)) SE <- NestsResult$SE
    
    if (is.null(resultmcmc) & (GTRN.CI=="mcmc")) {
      GTRN.CI <- NULL
    }
    if ((is.null(SE)) & (GTRN.CI == "se")) {
      GTRN.CI <- NULL
    }
    if ((is.null(hessian)) & (GTRN.CI == "hessian")) {
      GTRN.CI <- NULL
    }
    
    # if(all(is.na(SE))) replicate.CI <- 1
    
    nbtp <- length(temperatures)
    
    result.out <- universalmclapply(seq(from=1, to=nbtp-2, by=skip), 
                                    FUN=function(temp) {
                                      
                                      dt <- floor(as.numeric(times[temp:nbtp]-times[temp]))
                                      
                                      tempencours <- temperatures[temp:(temp+length(dt)-1)]
                                      if (metabolic.heating == 0) {
                                        tempnid2 <- temperatures.end.incubation[temp:(temp+length(dt)-1)]
                                        # et dans dt j ai le temps
                                        tempencours <- tempencours + (tempnid2-tempencours)*(dt/average.incubation.duration)
                                      }
                                      df <- data.frame(Time=dt, temp=tempencours)
                                      formated <- FormatNests(df)
                                      
                                      out.incubation <- info.nests(parameters=parameters, fixed.parameters = fixed.parameters,
                                                                   SE=SE, temperatures = formated, derivate=derivate, test=test, 
                                                                   stopattest = TRUE, M0=M0, TSP.borders=TSP.borders, 
                                                                   GTRN.CI=GTRN.CI, 
                                                                   hessian = hessian, 
                                                                   SexualisationTRN.CI=SexualisationTRN.CI, 
                                                                   embryo.stages=embryo.stages, 
                                                                   metabolic.heating=metabolic.heating, 
                                                                   temperature.heterogeneity=temperature.heterogeneity, 
                                                                   SexualisationTRN=SexualisationTRN, SexualisationTRN.mcmc=SexualisationTRN.mcmc, 
                                                                   progress = FALSE, 
                                                                   replicate.CI=replicate.CI, out="summary", warnings=FALSE, 
                                                                   resultmcmc=resultmcmc, parallel=FALSE)$summary
                                      
                                      # metric.end.incubation=c(Temp=test$Mean),
                                      return(cbind(Time=times[temp], out.incubation, row.names=NULL))
                                    }, 
                                    mc.cores = ifelse(parallel, detectCores(), 1), 
                                    clusterEvalQ=expression(library(embryogrowth)), 
                                    clusterExport=list(varlist=c("nbtp", "times", 
                                                                 "temperatures", "metabolic.heating", 
                                                                 "temperatures.end.incubation", "average.incubation.duration", 
                                                                 "resultmcmc", "replicate.CI", "parameters", "fixed.parameters", 
                                                                 "SE", "derivate", "test", "M0", "TSP.borders", "embryo.stages", 
                                                                 "temperature.heterogeneity", "SexualisationTRN", "skip"), 
                                                       envir = environment()), 
                                    progressbar = progressbar)
    
    
    df <- data.frame(matrix(unlist(result.out), ncol=41, byrow=TRUE))
    colnames(df) <- colnames(result.out[[1]])
    
    return(df)
    
  }
