#' predict.tsd Estimate sex ratio according to constant incubation temperature
#' @title Estimate sex ratio according to constant incubation temperature
#' @author Marc Girondot
#' @return A list with informations about sex-ratio
#' @param object A result file generated by tsd
#' @param temperatures A vector of temperatures
#' @param durations A vector of durations
#' @param range.CI The range of confidence interval for estimation, default=0.95
#' @param ... Not used
#' @description Estimate sex ratio according to constant incubation temperature
#' The list has the following components:
#' $sexratio $CI.minus.sexratio $CI.plus.sexratio $range.CI
#' if range.CI is set to NULL, the confidence interval is not estimated
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' m <- c(10, 14, 7, 4, 3, 0, 0)
#' f <- c(0, 1, 2, 4, 15, 10, 13)
#' t <- c(25, 26, 27, 28, 29, 30, 31)
#' result <- tsd(males=m, females=f, temperatures=t)
#' predict(result, temperatures=c(25, 31))
#' }
#' @method predict tsd
#' @export


predict.tsd <-
function(object, temperatures=NULL, durations=NULL,
	range.CI=0.95, ...) {
  
  if (is.null(temperatures) & !is.null(durations)) temperatures <- durations
  if (is.null(temperatures)) temperatures <- object$temperatures
   
  x <- object
	equation <- x$equation
	par <- x$par
	res <- x$SE
	pS <- 1/par["S"]
	pm <- NULL
  pp <- NULL
  	
if (is.null(range.CI)) {
  warning("The parameter range.CI cannot be NULL; it is changed to 0.95")
  range.CI <- 0.95
}
    range.CI.qnorm <- qnorm(1-((1-range.CI)/2))

	if (equation=="logistic") {
	  parameter.expand <- expand.grid(S=c(ifelse(sign(par["S"]-range.CI.qnorm*res["S"])!=sign(par["S"]), (sign(par["S"])*1e-5), (par["S"]-range.CI.qnorm*res["S"])), 
	                                      ifelse(sign(par["S"]+range.CI.qnorm*res["S"])!=sign(par["S"]), (sign(par["S"])*1e-5), (par["S"]+range.CI.qnorm*res["S"]))),
	                                  P=c(par["P"]-range.CI.qnorm*res["P"], par["P"]+range.CI.qnorm*res["P"]), 
                                    temperatures=temperatures)
	  y <- as.numeric(with(parameter.expand, 1/(1+exp(1/S*(P-temperatures)))))
	}
  
	if (equation=="Hill") {
	  # je mets 1/S dans le tableau
	  parameter.expand <- expand.grid(S=c(ifelse(sign(par["S"]-range.CI.qnorm*res["S"])!=sign(par["S"]), 1/(sign(par["S"])*1e-5), 1/(par["S"]-range.CI.qnorm*res["S"])), 
	                                      ifelse(sign(par["S"]+range.CI.qnorm*res["S"])!=sign(par["S"]), 1/(sign(par["S"])*1e-5), 1/(par["S"]+range.CI.qnorm*res["S"]))),
	                                  P=c(par["P"]-range.CI.qnorm*res["P"], par["P"]+range.CI.qnorm*res["P"]), temperatures=temperatures)
	  y <- as.numeric(with(parameter.expand, 1/(1+exp(S*(log(P)-log(temperatures))))))
	}
	
	if (equation=="Richards") {
	  # attention, ici je laisse S
	  parameter.expand <- expand.grid(S=c(ifelse(sign(par["S"]-range.CI.qnorm*res["S"])!=sign(par["S"]), sign(par["S"])*1e-5, par["S"]-range.CI.qnorm*res["S"]), 
	                                      ifelse(sign(par["S"]+range.CI.qnorm*res["S"])!=sign(par["S"]), sign(par["S"])*1e-5, par["S"]+range.CI.qnorm*res["S"])),
	                                  P=c(par["P"]-range.CI.qnorm*res["P"], par["P"]+range.CI.qnorm*res["P"]), 
	                                  K=c(ifelse(sign(par["K"]-range.CI.qnorm*res["K"])!=sign(par["K"]), sign(par["K"])*1e-5, par["K"]-range.CI.qnorm*res["K"]), 
	                                      ifelse(sign(par["K"]+range.CI.qnorm*res["K"])!=sign(par["K"]), sign(par["K"])*1e-5, par["K"]+range.CI.qnorm*res["K"])), 
	                                  temperatures=temperatures)
	  y <- as.numeric(with(parameter.expand, 
	                       (1+(2^exp(K)-1)*exp((1/S)*(P-temperatures)))^(-1/exp(K))
	  ))
	}
  
	if (equation=="Double-Richards") {
	  # attention, ici je laisse S
	  parameter.expand <- expand.grid(S=c(ifelse(sign(par["S"]-range.CI.qnorm*res["S"])!=sign(par["S"]), sign(par["S"])*1e-5, par["S"]-range.CI.qnorm*res["S"]), 
	                                      ifelse(sign(par["S"]+range.CI.qnorm*res["S"])!=sign(par["S"]), sign(par["S"])*1e-5, par["S"]+range.CI.qnorm*res["S"])),
	                                  P=c(par["P"]-range.CI.qnorm*res["P"], par["P"]+range.CI.qnorm*res["P"]), 
	                                  K1=c(ifelse(sign(par["K1"]-range.CI.qnorm*res["K1"])!=sign(par["K1"]), sign(par["K1"])*1e-5, par["K1"]-range.CI.qnorm*res["K1"]), 
	                                       ifelse(sign(par["K1"]+range.CI.qnorm*res["K1"])!=sign(par["K1"]), sign(par["K1"])*1e-5, par["K1"]+range.CI.qnorm*res["K1"])), 
	                                  K2=c(ifelse(sign(par["K2"]-range.CI.qnorm*res["K2"])!=sign(par["K2"]), sign(par["K2"])*1e-5, par["K2"]-range.CI.qnorm*res["K2"]), 
	                                       ifelse(sign(par["K2"]+range.CI.qnorm*res["K2"])!=sign(par["K2"]), sign(par["K2"])*1e-5, par["K2"]+range.CI.qnorm*res["K2"])), 
	                                  temperatures=temperatures)
	  y <- as.numeric(with(parameter.expand, ifelse(temperatures<P, 
	                                                ifelse(K1>3 & sign(P-temperatures)==sign(S), 
	                                                       0.5*exp((temperatures-P)/(S*exp(K1))), 
	                                                       (1+(2^exp(K1)-1)*exp((1/S)*(P-temperatures)))^(-1/exp(K1))),
	                                                ifelse(K2>3 & sign(P-temperatures)==sign(S), 
	                                                       0.5*exp((temperatures-P)/(S*exp(K2))), 
	                                                       (1+(2^exp(K2)-1)*exp((1/S)*(P-temperatures)))^(-1/exp(K2)))
	                                                
	  )
	  ))
	}
	
  
	if (equation=="Hulin") {
    # attention, ici je laisse S
	  parameter.expand <- expand.grid(S=c(ifelse(sign(par["S"]-range.CI.qnorm*res["S"])!=sign(par["S"]), sign(par["S"])*1e-5, par["S"]-range.CI.qnorm*res["S"]), 
	                                      ifelse(sign(par["S"]+range.CI.qnorm*res["S"])!=sign(par["S"]), sign(par["S"])*1e-5, par["S"]+range.CI.qnorm*res["S"])),
	                                  P=c(par["P"]-range.CI.qnorm*res["P"], par["P"]+range.CI.qnorm*res["P"]), 
	                                  K1=c(ifelse(sign(par["K1"]-range.CI.qnorm*res["K1"])!=sign(par["K1"]), sign(par["K1"])*1e-5, par["K1"]-range.CI.qnorm*res["K1"]), 
	                                       ifelse(sign(par["K1"]+range.CI.qnorm*res["K1"])!=sign(par["K1"]), sign(par["K1"])*1e-5, par["K1"]+range.CI.qnorm*res["K1"])), 
	                                  K2=c(ifelse(sign(par["K2"]-range.CI.qnorm*res["K2"])!=sign(par["K2"]), sign(par["K2"])*1e-5, par["K2"]-range.CI.qnorm*res["K2"]), 
	                                       ifelse(sign(par["K2"]+range.CI.qnorm*res["K2"])!=sign(par["K2"]), sign(par["K2"])*1e-5, par["K2"]+range.CI.qnorm*res["K2"])), 
	                                  temperatures=temperatures)
	  y <- as.numeric(with(parameter.expand, 
	                       (1+(2^exp(K1*temperatures+K2)-1)*exp((1/S)*(P-temperatures)))^(-1/exp(K1*temperatures+K2))
	  ))
	}

# si des températures sont égales, elles sont regroupées or je les veux séparées
# nrow(parameter.expand)/length(temperatures) répétitions
fac <- NULL
for (i in 1:length(temperatures))
  fac <- c(fac, rep(as.character(i), nrow(parameter.expand)/length(temperatures)))

l <- split(y, fac)
ordre <- order(as.numeric(names(l)))

pm <- unlist(lapply(l, min))[ordre]
pp <- unlist(lapply(l, max))[ordre]




		if (equation=="GSD") {
			p <- rep(0.5, length(temperatures))
		}
	
		if (equation=="logistic") {
  			p <- 1/(1+exp(pS*(par["P"]-temperatures)))
		}

		if (equation=="Richards") {
			pK1 <- (2^exp(par["K"])-1)
			pK2 <- (-1/exp(par["K"]))
			pKS <- (par["S"]*exp(par["K"]))
			p <- ifelse(par["K"]>3 & sign(par["P"]-temperatures)==sign(par["S"]), 
			0.5*exp((temperatures-par["P"])/pKS), 
			(1+pK1*exp(pS*(par["P"]-temperatures)))^pK2)
		}

if (equation=="Double-Richards") {
  pK1_1 <- (2^exp(par["K1"])-1)
  pK2_1 <- (-1/exp(par["K1"]))
  pKS_1 <- (par["S"]*exp(par["K1"]))
  pK1_2 <- (2^exp(par["K2"])-1)
  pK2_2 <- (-1/exp(par["K2"]))
  pKS_2 <- (par["S"]*exp(par["K2"]))
  
  p <- ifelse(temperatures<par["P"],
    ifelse(par["K1"]>3 & sign(par["P"]-temperatures)==sign(par["S"]), 
              0.5*exp((temperatures-par["P"])/pKS_1), 
              (1+pK1_1*exp(pS*(par["P"]-temperatures)))^pK2_1),
    ifelse(par["K2"]>3 & sign(par["P"]-temperatures)==sign(par["S"]), 
           0.5*exp((temperatures-par["P"])/pKS_2), 
           (1+pK1_2*exp(pS*(par["P"]-temperatures)))^pK2_2)
  )
}


  if (equation=="Hulin") {
    p <- (1+(2^exp(par["K1"]*temperatures+par["K2"])-1)*exp(pS*(par["P"]-temperatures)))^(-1/exp(par["K1"]*temperatures+par["K2"]))
  }

		if (equation=="Hill") {
  			p <- 1/(1+exp(pS*(log(par["P"])-log(temperatures))))
		}

 # 		p <- ifelse(p==0, 1E-9, p)
 # 		p <- ifelse(p==1, 1-1E-9, p)
	if (x$males.freq) {
	  return(list(sexratio=as.numeric(p), CI.minus.sexratio=as.numeric(pm), CI.plus.sexratio=as.numeric(pp), range.CI=range.CI))
	} else {
	  return(list(sexratio=as.numeric(1-p), CI.minus.sexratio=as.numeric(1-pp), CI.plus.sexratio=as.numeric(1-pm), range.CI=range.CI))
	}

}
