% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generate-EICM.r
\name{generateEICM}
\alias{generateEICM}
\title{Generate EICM model following a Beta frequency distribution}
\usage{
generateEICM(nspecies, nsamples, nenv, ninteractions, shape1, shape2,
  nbins = 10, nrepetitions = 5, shrinkage = 2, bounds = 10,
  swarm.size = floor((ninteractions + nspecies * (nenv + 1)) * 0.5),
  maxit.stagnate = 150)
}
\arguments{
\item{nspecies}{the number of species to generate.}

\item{nsamples}{the number of samples to generate.}

\item{nenv}{the number of environmental predictors to generate.}

\item{ninteractions}{the number of species interactions to generate.}

\item{shape1}{the shape1 parameter of the Beta distribution.}

\item{shape2}{the shape2 parameter of the Beta distribution.}

\item{nbins}{the number of histogram bins for comparing distributions.}

\item{nrepetitions}{the number of times to realize a model candidate to average their distributions.}

\item{shrinkage}{the shrinkage factor for generated coefficients, when computing fitness criterion.
Ensures that the generated coefficients remain at plausible values.}

\item{bounds}{the allowed range for the coefficients \code{c(-bounds, +bounds)}.}

\item{swarm.size}{the swarm size of the particle swarm optimization.}

\item{maxit.stagnate}{the number of iterations without improvement until the optimization stops.}
}
\value{
A EICM model of class \code{eicm}
}
\description{
Generates a randomly parameterized EICM model (predictors, environmental coefficients and species interactions),
ensuring that communities simulated with this model have a frequency distribution that matches the given Beta
distribution of frequencies as much as possible.
}
\details{
This function is useful for generating a realistic random model for simulation, i.e. a model that, when simulated, will
yield species communities with a distribution of frequencies akin of real communities: with most species being rare.
The generated coefficients are not assumed to follow any distribution, but are optionally shrinked so that
their values will remain "decent". The values of the environmental predictors are drawn from a gaussian distribution.
}
\examples{
\donttest{
# Generate model with 32 species, 30 species interactions and 2 environmental predictors
# for 500 samples with a frequency distribution following a Beta(1.5, 3)
model <- generateEICM(nspecies=32, nsamples=500, nenv=2, ninteractions=30,
  shape1=1.5, shape2=3)

# make one realization
data <- predict(model, nrepetitions=1)

# plot frequency histogram: should follow a Beta distribution.
hist(apply(data, 2, sum) / nrow(data), breaks=seq(0, 1, by=0.1), xlim=c(0, 1),
  main="Frequency distribution of one realization", xlab="Frequency in samples",
  ylab="Number of species")
}
}
