% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extend.R, R/generic_functions.R
\name{extend.cord}
\alias{extend.cord}
\alias{extend}
\title{Simulate or extend multivariate abundance data}
\usage{
\method{extend}{cord}(
  object,
  N = nrow(object$obj$data),
  coeffs = coef(object$obj),
  newdata = NULL,
  n_replicate = NULL,
  do.fit = FALSE,
  seed = NULL
)

extend(
  object,
  N = nrow(object$obj$data),
  coeffs = coef(object$obj),
  newdata = NULL,
  n_replicate = NULL,
  do.fit = FALSE,
  seed = NULL
)
}
\arguments{
\item{object}{objects of class \code{cord}, typically the result of a call to \code{\link[ecoCopula]{cord}}.}

\item{N}{Number of samples to be extended. Defaults to the number of observations in the original sample.}

\item{coeffs}{Coefficient matrix for a \code{\link[mvabund]{manyglm}} object that characterises the size of effects to be simulated.
See \code{\link{effect_alt}} for help in producing this matrix. Defaults to the coefficient matrix from the \code{\link[ecoCopula]{cord}}
object, \code{coef(object$obj)}.}

\item{newdata}{Data frame of same size as the original X covariates from the fitted \code{object}, that specifies
a different design of interest. Defaults to \code{NULL}.}

\item{n_replicate}{Number of unique replicates of the original data frame. Defaults to \code{NULL}, overwrites \code{N} if specified.}

\item{do.fit}{Logical. If \code{TRUE}, fits a \code{\link[mvabund]{manyglm}} object from the simulated data. Defaults to \code{FALSE}.}

\item{seed}{Random number seed, defaults to a random seed number.}
}
\value{
Simulated data or \code{manyglm} object.
}
\description{
\code{extend} returns a simulated response matrix or a \code{\link[mvabund]{manyglm}} object with \code{N} observations
and simulated response matrix that utilises the existing correlation structure of the data.
}
\details{
\code{extend} takes a \code{\link[ecoCopula]{cord}} object and returns a new simulated response matrix or an "extended" \code{\link[mvabund]{manyglm}} object
with \code{N} observations and the new simulated response matrix. Response abundances are simulated through a Gaussian
copula model that utilises a coefficient matrix \code{coeffs}, the specified \code{cord} model and the joint
correlation structure exhibited between the response variables. To help with the specification of
\code{coeffs}, see \code{\link{effect_alt}} which simplifies this process.

Response variables are simulated through a copula model by first extracting Gaussian copular scores
as Dunn-Smyth residuals (Dunn & Smyth 1996), which are obtained from abundances \eqn{y_{ij}} with marginal distributions
\eqn{F_j} which have been specified via the original \code{manyglm} model (\code{fit.glm}; see examples);

\deqn{z_{ij} = \Phi^{-1}{F_{j}(y_{ij}^-) + u_{ij} f_{j}(y_{ij})}}

 These scores then follow a multivariate Gaussian distribution with zero mean and covariance structure \eqn{\Sigma},

 \deqn{z_{ij} \sim N_p(0,\Sigma)}

To avoid estimating a large number \eqn{p(p-1)/2} pairwise correlations within \eqn{\Sigma}, factor analysis is utilised
with two latent factor variables, which can be interpreted as an unobserved environmental covariate.

Thus, in order to simulate new multivariate abundances we simulate new copula scores and back transform them to
abundances as \eqn{y_{ij}= {F^*}_j^{-1}(\Phi(z_{ij}))}, where the coefficient matrix \code{coeffs} specifies the
effect size within the new marginal distributions \eqn{{F^*}_j}.

The data frame is also extended in a manner that preserves the original design structure. This is done by first
repeating the design matrix until the number of samples exceeds \code{N}, then randomly removing rows from the last
repeated data frame until the number of samples equals \code{N}. Alternatively, a balanced design structure can be
obtained by specifying the number of replicates.

\code{newdata} can be utilised if a different data frame is wanted for simulation.

If users are interested in obtaining a \code{manyglm} model, \code{do.fit=TRUE} can be used to obtain a \code{\link[mvabund]{manyglm}}
object from the simulated responses.
}
\section{Functions}{
\itemize{
\item \code{extend}: Simulate or extend multivariate abundance data
}}

\examples{
library(ecoCopula)
library(mvabund)
data(spider)
spiddat = mvabund(spider$abund)
X = data.frame(spider$x)

# Specify increasers and decreasers
increasers = c("Alopacce", "Arctlute", "Arctperi", "Pardnigr", "Pardpull")
decreasers = c("Alopcune", "Alopfabr", "Zoraspin")

# Simulate data
fit.glm = manyglm(spiddat~1, family="negative.binomial")
fit.cord = cord(fit.glm)
simData = extend(fit.cord)

# Simulate data with N=20
fit.glm = manyglm(spiddat~soil.dry, family="negative.binomial", data=X)
fit.cord = cord(fit.glm)
simData = extend(fit.cord, N=20)

# Obtain a manyglm fit from simulated data with N=10 and effect_size=1.5
X$Treatment = rep(c("A","B","C","D"),each=7)
fit_factors.glm = manyglm(spiddat~Treatment, family="negative.binomial", data=X)
effect_mat = effect_alt(fit_factors.glm, effect_size=1.5,
     increasers, decreasers, term="Treatment")
fit_factors.cord = cord(fit_factors.glm)
newFit.glm = extend(fit_factors.cord, N=10,
     coeffs=effect_mat, do.fit=TRUE)

# Change sampling design
X_new = X
X_new$Treatment[6:7] = c("B","B")
simData = extend(fit_factors.cord, N=NULL,
   coeffs=effect_mat, newdata=X_new, n_replicate=5)
}
\references{
Dunn, P.K., & Smyth, G.K. (1996). Randomized quantile residuals. Journal of Computational and Graphical Statistics 5, 236-244.
}
\seealso{
\code{\link{effect_alt}}
}
