\name{repeat.sample}
\alias{repeat.sample}
\title{Generates OLS Data and Confidence/Prediction Intervals for Repeated Samples}
\usage{
repeat.sample(
  x,
  true.par,
  omit = 0,
  mean = 0,
  sd = 1,
  rep = 100,
  xnew = x,
  sig.level = 0.05,
  seed = NULL
)
}
\arguments{
\item{x}{(n x k) vector or matrix of exogenous data, where each column represents the data of one of k exogenous predictors. The number of rows represents the sample size n.}

\item{true.par}{vector of true parameters in the linear model (level and slope parameters). If \code{true.par} is a vector without named elements then coefficients are named "alpha", "beta1", "beta2", ..., "betak" by default. Otherwise the names specified are used.}

\item{omit}{vector of indices identifying the exogenous variables to be omitted in the true model, e.g. \code{omit = 1} corresponds to the first exogenous variable to be omitted. This argument can be used to illustrate omitted variable bias in parameter and standard error estimates. Default value is \code{omit = 0}, i.e. no exogenous variable is omitted}

\item{mean}{expected value of the normal distribution of the error term.}

\item{sd}{standard deviation of the normal distribution of the error term. Used only for generating simulated y-values. Interval estimators use the estimated sigma.}

\item{rep}{repetitions, i.e. number of simulated samples. The samples in each matrix generated have enumerated names "SMPL1", "SMPL2", ..., "SMPLs".}

\item{xnew}{(t x k) matrix of new exogenous data points at which prediction intervals should be calculated. t corresponds to the number of new data points, k to the number of exogenous variables in the model. If not specified regular values \code{x} are used (see first argument).}

\item{sig.level}{significance level for confidence and prediction intervals.}

\item{seed}{optionally set random seed to arbitrary number if results should be made replicable.}
}
\value{
A list of named data structures. Let s = number of samples, n = sample size, k = number of coefficients, t = number of new data points in \code{xnew} then:
\tabular{ll}{
\code{x} \tab (n x k matrix): copy of data of exogenous regressors that was passed to the function.\cr
\code{y} \tab (n x s matrix): simulated real y values in each sample.\cr
\code{fitted} \tab	(n x s matrix): estimated y values in each sample.\cr
\code{coef} \tab (k x s matrix): estimated parameters in each sample.\cr
\code{true.par} \tab	(k vector): vector of true parameter values (implemented only for \code{plot.confint()}).\cr
\code{u} \tab (n x s matrix): random error term in each sample.\cr
\code{residuals} \tab (n x s matrix): residuals of OLS estimations in each sample.\cr
\code{sig.squ} \tab (s vector): estimated variance of the error term in each sample.\cr
\code{var.u} \tab (s vector): variance of random errors drawn in each sample.\cr
\code{se} \tab (k x s matrix): estimated standard deviation of the coefficients in each sample.\cr
\code{vcov.coef} \tab (k x k x s array): estimated variance-covariance matrix of the coefficients in each sample.\cr
\code{confint} \tab (k x 2 x s array): confidence intervals of the coefficients in each sample. Interval bounds are named "lower" and "upper".\cr
\code{outside.ci} \tab (k vector): percentage of confidence intervals not covering the true value for each of the regression parameters.\cr
\code{y0} \tab	(t x s matrix): simulated real future y values at \code{xnew} in each sample (real line plus real error).\cr
\code{y0.fitted} \tab (t x s matrix): point prediction, i.e. estimated y values at \code{xnew} in each sample (regression line).\cr
\code{predint} \tab (t x 2 x s array): prediction intervals of future endogenous realizations at exogenous data points specified by \code{xnew}. Intervals are calculated for each sample, respectively. Interval bounds are named "lower" and "upper".\cr
\code{sd.pe} \tab (t x s matrix): estimated standard deviation of prediction errors at all exogenous data points in each sample.\cr
\code{outside.pi} \tab	(t vector): percentage of prediction intervals not covering the true value \code{y0} at \code{xnew}.\cr
\code{bias.coef} \tab (k vector): true bias in parameter estimators if variables are omitted (argument \code{omit} unequal to zero).\cr
}
}
\description{
This command simulates repeated samples given fixed data of the exogenous predictors and given (true) regression parameters. For each sample generated the results from an OLS regression with level parameter and confidence intervals (CIs) as well as prediction intervals are calculated.
}
\details{
Let \code{X} be an object generated by \code{repeat.sample()} then \code{plot(X, ...)} accepts the following arguments:
\tabular{ll}{
\code{plot.what = "confint"} \tab plot stacked confidence intervals for all samples. Additional arguments are \code{center = TRUE} (plot center of intervals?), \code{which.coef = 2} (intervals for which coefficient?), \code{center.size = 1} (size of the center dot), \code{lwd = 1} (line width).\cr
\code{plot.what = "reglines"} \tab plot regression lines of all samples.\cr
\code{plot.what = "scatter"} \tab plot scatter plots of all samples.\cr
}
}
\examples{
## Generate data of two predictors
x1 = c(1,2,3,4,5)
x2 = c(2,4,5,5,6)
x = cbind(x1,x2)

## Generate list of data structures and name it "out"
out = repeat.sample(x, true.par = c(2,1,4), rep = 10)

## Extract some data
out$coef[2,8] # Extract estimated beta1 (i.e. 2nd coef) in the 8th sample
out$coef["beta1","SMPL8"] # Same as above using internal names
out$confint["beta1","upper","SMPL5"] # Extract only upper bound of CI of beta 1 from 5th sample
out$confint[,,5] # Extract CIs (upper and lower bound) for all parameters from 5th sample
out$confint[,,"SMPL5"] # Same as above using internal names
out$confint["beta1",,"SMPL5"] # Extract CI of beta 1 from 5th sample
out$u.hat[,"SMPL7"] # Extract residuals from OLS estimation of sample 7

## Generate prediction intervals at three specified points of exogenous data (xnew)
out = repeat.sample(x, true.par = c(2,1,4), rep = 10,
      xnew = cbind(x1 = c(1.5,6,7), x2 = c(1,3,5.5)))
out$predint[,,6] # Prediction intervals at the three data points of xnew in 6th sample
out$sd.pe[,6] # Estimated standard deviations of prediction errors in 6th sample
out$outside.pi # Percentage of how many intervals miss true y0 realization

## Illustrate that the relative shares of cases when the interval does not cover the
## true value approaches the significance level
out = repeat.sample(x, true.par = c(2,1,4), rep = 1000)
out$outside.ci

## Illustrate omitted variable bias
out.unbiased = repeat.sample(x, true.par = c(2,1,4))
mean(out.unbiased$coef["beta1",]) # approx. equal to beta1 = 1
out.biased = repeat.sample(x, true.par = c(2,1,4), omit = 2) # omit x2
mean(out.biased$coef["beta1",]) # not approx. equal to beta1 = 1
out.biased$bias.coef # show the true bias in coefficients

## Simulate a regression with given correlation structure in exogenous data
corr.mat = cbind(c(1, 0.9),c(0.9, 1)) # Generate desired corr. structure (high autocorrelation)
X = makedata.corr(n = 10, k = 2, CORR = corr.mat) # Generate 10 obs. of 2 exogenous variables
out = repeat.sample(X, true.par = c(2,1,4), rep = 1) # Simulate a regression
out$vcov.coef

## Illustrate confidence intervals
out = repeat.sample(c(10, 20, 30,50), true.par = c(0.2,0.13), rep = 10, seed = 12)
plot(out, plot.what = "confint")

## Plots confidence intervals of alpha with specified \code{xlim} values.
plot(out, plot.what = "confint", which.coef = 1, xlim = c(-15,15))

## Illustrate normality of dependent variable
out = repeat.sample(c(10,30,50), true.par = c(0.2,0.13), rep = 200)
plot(out, plot.what = "scatter")

## Illustrate confidence bands in a regression
plot(out, plot.what = "reglines")

}
\concept{CI}
\concept{confidence interval}
\concept{omitted variable bias}
\concept{ordinary least squares}
\concept{prediction interval}
\concept{repeated samples}
\concept{simulation}
\concept{true regression parameters}
