% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/deFit.R
\name{defit}
\alias{defit}
\title{Fitting Differential Equations to Time Series Data}
\usage{
defit(data, model, guess = NULL, method = NULL, plot = FALSE)
}
\arguments{
\item{data}{a data frame containing all model variables. The "time" column must be included.}

\item{model}{a string specifying the model to be used. The "=~" operator is used to define variables, with the name of the variable user defined on the left and the name of the variable in the data on the right. The '~' operator specifies a differential equation, with the dependent variable on the left and the independent variables on the right. See also ‘Details’.}

\item{guess}{an optional vector that allows the user to give starting values for the model parameters, including the model coefficients and variable initial states.}

\item{method}{an optional string indicating which optimizer to use. The default method is subject to the specific model. The available options are 'Nelder-Mead','L-BFGS-B','SANN' and 'BFGS'.}

\item{plot}{an optional TRUE or FALSE that TRUE will draw the plot of the raw data and the predicted values.}
}
\value{
object: directly type the defit object will print all results. The function summary is used to print the summary of all results, and the exact values of each result can be extracted by the "$" operator.

userdata: the data that contains a sequence 'seq' starting from 1, the original time variable 'time', and all other variables user defined.

parameter: the best set of parameters found, including parameter values, gradient, convergence, message and hessian matrix.

predict: a dataframe of model predicted variable states at each time point.

r_squared: r_squared is the square of the correlation between the observed values and the predicted values, representing the proportion of variance explained by the model.

RMSE: RMSE (Root Mean Squared Error) is the standard deviation of the residuals.

SE: a symmetric matrix giving standard error of the model parameters.

equation: a string prints the estimated differential equations and initial states.

table: a summary table of parameter estimates and their corresponding SEs.

convergence: a message returns the result of the optimization convergence check.
}
\description{
Use numerical optimization to fit ordinary differential equations (ODEs) to time series data to examine the dynamic relationships between variables or the characteristics of a dynamical system. It can now be used to estimate the parameters of ODEs up to second order, and can also apply to multilevel systems.
}
\details{
We suggest choosing the method by default. The guess values contain the coefficient of the model and initial values (the values of t0). Different models have different number of values.

Time(param) sequence for which output is wanted; the first value of times must be the initial time.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# eg1. An example of the univariate second-order differential equation (damped oscillator model)
data('example1')
model1 <- '
   X =~ myX
   time =~ myTime
   X(2) ~ X(1) + X
  '
result1 <- defit(data = example1, model = model1)
#> Program will fit the data with a univariate second-order differential equation.
#> The differential equation is:
#> x(2) = beta1 * x + beta2 * x(1)
#> Optimizing...
#> Finishing optimization...
#> Estimating R_squared
#> Estimating RMSE
#> Estimating Hessian
# result1$table get the result
# names(result1) get all names of object

#--------------
# eg3. An example of the multilevel univariate second-order differential equation
data('example3')
model3 <- '
   X =~ current
   time =~ myTime
   X(2) ~ X(1) + X + (1 + X(1) + X | year)
   '
example3_use <- example3[(example3["year"] >= 2015)&(example3["year"] <= 2018),] # Note: select a subset of the data as an example.
example3_c <- scale_within(example3_use, model3) # note: centering X variable by year
result3 <- defit(data=example3_c,model = model3,plot=FALSE)
#> subject checking
#> Random effects and fixed effects have been defined
#> Estimating population parameter 
#> Waiting...
#> next...subject...2015
#> next...subject...2016
#> next...subject...2017
#> next...subject...2018
#> Estimating R_squared
#> Estimating RMSE
#> Estimating Hessian

#--------------
# eg4. An example of the multilevel bivariate first-order differential equations
data('example3')
model4 <- '
   X =~ current
   Y =~ expected
   time =~ myTime
   X(1) ~ X + Y + (1 + X + Y | year)
   Y(1) ~ X + Y + (1 + X + Y | year)
   '
example4_use <- example3[(example3["year"] >= 2015)&(example3["year"] <= 2018),] # Note: select a subset of the data as an example.
example4_c <- scale_within(example4_use, model4) # centering X and Y variable by year
result4 <- defit(data=example4_c,model = model4,plot=FALSE)
#> subject checking
#> Random effects and fixed effects have been defined
#> Estimating parameters 
#> Waiting...
#> next...subject...2015
#> next...subject...2016
#> next...subject...2017
#> next...subject...2018
#> Estimating R_squared
#> Estimating RMSE
#> Estimating Hessian
}\if{html}{\out{</div>}}
}
\examples{
#eg2. An example of bivariate first-order differential equation
data('example2')
model2 <- '
    # define variable
    X =~ myX
    Y =~ myY

    # define time
    time =~ myTime

    # define differential equation
    X(1) ~ X + Y
    Y(1) ~ Y + X
  '
result2 <- defit(data = example2, model = model2)
result2
# extract details and values
# result2$summary()
# result2$userdata
# result2$parameter$par
# result2$equation
# result2$table
# result2$plot()

}
