% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data_tabulate.R
\name{data_tabulate}
\alias{data_tabulate}
\alias{data_tabulate.default}
\alias{data_tabulate.data.frame}
\alias{as.data.frame.datawizard_tables}
\title{Create frequency and crosstables of variables}
\usage{
data_tabulate(x, ...)

\method{data_tabulate}{default}(
  x,
  by = NULL,
  drop_levels = FALSE,
  weights = NULL,
  remove_na = FALSE,
  proportions = NULL,
  name = NULL,
  verbose = TRUE,
  ...
)

\method{data_tabulate}{data.frame}(
  x,
  select = NULL,
  exclude = NULL,
  ignore_case = FALSE,
  regex = FALSE,
  by = NULL,
  drop_levels = FALSE,
  weights = NULL,
  remove_na = FALSE,
  proportions = NULL,
  collapse = FALSE,
  verbose = TRUE,
  ...
)

\method{as.data.frame}{datawizard_tables}(
  x,
  row.names = NULL,
  optional = FALSE,
  ...,
  stringsAsFactors = FALSE,
  add_total = FALSE
)
}
\arguments{
\item{x}{A (grouped) data frame, a vector or factor.}

\item{...}{not used.}

\item{by}{Optional vector or factor. If supplied, a crosstable is created.
If \code{x} is a data frame, \code{by} can also be a character string indicating the
name of a variable in \code{x}.}

\item{drop_levels}{Logical, if \code{FALSE}, factor levels that do not occur in
the data are included in the table (with frequency of zero), else unused
factor levels are dropped from the frequency table.}

\item{weights}{Optional numeric vector of weights. Must be of the same length
as \code{x}. If \code{weights} is supplied, weighted frequencies are calculated.}

\item{remove_na}{Logical, if \code{FALSE}, missing values are included in the
frequency or crosstable, else missing values are omitted.}

\item{proportions}{Optional character string, indicating the type of
percentages to be calculated. Only applies to crosstables, i.e. when \code{by} is
not \code{NULL}. Can be \code{"row"} (row percentages), \code{"column"} (column percentages)
or \code{"full"} (to calculate relative frequencies for the full table).}

\item{name}{Optional character string, which includes the name that is used
for printing.}

\item{verbose}{Toggle warnings.}

\item{select}{Variables that will be included when performing the required
tasks. Can be either
\itemize{
\item a variable specified as a literal variable name (e.g., \code{column_name}),
\item a string with the variable name (e.g., \code{"column_name"}), or a character
vector of variable names (e.g., \code{c("col1", "col2", "col3")}),
\item a formula with variable names (e.g., \code{~column_1 + column_2}),
\item a vector of positive integers, giving the positions counting from the left
(e.g. \code{1} or \code{c(1, 3, 5)}),
\item a vector of negative integers, giving the positions counting from the
right (e.g., \code{-1} or \code{-1:-3}),
\item one of the following select-helpers: \code{starts_with()}, \code{ends_with()},
\code{contains()}, a range using \code{:} or \code{regex("")}. \code{starts_with()},
\code{ends_with()}, and  \code{contains()} accept several patterns, e.g
\code{starts_with("Sep", "Petal")}.
\item or a function testing for logical conditions, e.g. \code{is.numeric()} (or
\code{is.numeric}), or any user-defined function that selects the variables
for which the function returns \code{TRUE} (like: \code{foo <- function(x) mean(x) > 3}),
\item ranges specified via literal variable names, select-helpers (except
\code{regex()}) and (user-defined) functions can be negated, i.e. return
non-matching elements, when prefixed with a \code{-}, e.g. \code{-ends_with("")},
\code{-is.numeric} or \code{-(Sepal.Width:Petal.Length)}. \strong{Note:} Negation means
that matches are \emph{excluded}, and thus, the \code{exclude} argument can be
used alternatively. For instance, \code{select=-ends_with("Length")} (with
\code{-}) is equivalent to \code{exclude=ends_with("Length")} (no \code{-}). In case
negation should not work as expected, use the \code{exclude} argument instead.
}

If \code{NULL}, selects all columns. Patterns that found no matches are silently
ignored, e.g. \code{extract_column_names(iris, select = c("Species", "Test"))}
will just return \code{"Species"}.}

\item{exclude}{See \code{select}, however, column names matched by the pattern
from \code{exclude} will be excluded instead of selected. If \code{NULL} (the default),
excludes no columns.}

\item{ignore_case}{Logical, if \code{TRUE} and when one of the select-helpers or
a regular expression is used in \code{select}, ignores lower/upper case in the
search pattern when matching against variable names.}

\item{regex}{Logical, if \code{TRUE}, the search pattern from \code{select} will be
treated as regular expression. When \code{regex = TRUE}, select \emph{must} be a
character string (or a variable containing a character string) and is not
allowed to be one of the supported select-helpers or a character vector
of length > 1. \code{regex = TRUE} is comparable to using one of the two
select-helpers, \code{select = contains("")} or \code{select = regex("")}, however,
since the select-helpers may not work when called from inside other
functions (see 'Details'), this argument may be used as workaround.}

\item{collapse}{Logical, if \code{TRUE} collapses multiple tables into one larger
table for printing. This affects only printing, not the returned object.}

\item{row.names}{\code{NULL} or a character vector giving the row
    names for the data frame.  Missing values are not allowed.}

\item{optional}{logical. If \code{TRUE}, setting row names and
    converting column names (to syntactic names: see
    \code{\link[base]{make.names}}) is optional.  Note that all of \R's
    \pkg{base} package \code{as.data.frame()} methods use
    \code{optional} only for column names treatment, basically with the
    meaning of \code{\link[base]{data.frame}(*, check.names = !optional)}.
    See also the \code{make.names} argument of the \code{matrix} method.}

\item{stringsAsFactors}{logical: should the character vector be converted
    to a factor?}

\item{add_total}{For crosstables (i.e. when \code{by} is not \code{NULL}), a row and
column with the total N values are added to the data frame. \code{add_total} has
no effect in \code{as.data.frame()} for simple frequency tables.}
}
\value{
A data frame, or a list of data frames, with one frequency table
as data frame per variable.
}
\description{
This function creates frequency or crosstables of variables,
including the number of levels/values as well as the distribution of raw,
valid and cumulative percentages. For crosstables, row, column  and cell
percentages can be calculated.
}
\details{
There is an \code{as.data.frame()} method, to return the frequency tables as a
data frame. The structure of the returned object is a nested data frame,
where the first column contains name of the variable for which frequencies
were calculated, and the second column is a list column that contains the
frequency tables as data frame. See 'Examples'.
}
\note{
There are \code{print_html()} and \code{print_md()} methods available for printing
frequency or crosstables in HTML and markdown format, e.g.
\code{print_html(data_tabulate(x))}.
}
\section{Crosstables}{

If \code{by} is supplied, a crosstable is created. The crosstable includes \verb{<NA>}
(missing) values by default. The first column indicates values of \code{x}, the
first row indicates values of \code{by} (including missing values). The last row
and column contain the total frequencies for each row and column, respectively.
Setting \code{remove_na = FALSE} will omit missing values from the crosstable.
Setting \code{proportions} to \code{"row"} or \code{"column"} will add row or column
percentages. Setting \code{proportions} to \code{"full"} will add relative frequencies
for the full table.
}

\examples{
\dontshow{if (requireNamespace("poorman")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# frequency tables -------
# ------------------------
data(efc)

# vector/factor
data_tabulate(efc$c172code)

# drop missing values
data_tabulate(efc$c172code, remove_na = TRUE)

# data frame
data_tabulate(efc, c("e42dep", "c172code"))

# grouped data frame
suppressPackageStartupMessages(library(poorman, quietly = TRUE))
efc \%>\%
  group_by(c172code) \%>\%
  data_tabulate("e16sex")

# collapse tables
efc \%>\%
  group_by(c172code) \%>\%
  data_tabulate("e16sex", collapse = TRUE)

# for larger N's (> 100000), a big mark is automatically added
set.seed(123)
x <- sample(1:3, 1e6, TRUE)
data_tabulate(x, name = "Large Number")

# to remove the big mark, use "print(..., big_mark = "")"
print(data_tabulate(x), big_mark = "")

# weighted frequencies
set.seed(123)
efc$weights <- abs(rnorm(n = nrow(efc), mean = 1, sd = 0.5))
data_tabulate(efc$e42dep, weights = efc$weights)

# crosstables ------
# ------------------

# add some missing values
set.seed(123)
efc$e16sex[sample.int(nrow(efc), 5)] <- NA

data_tabulate(efc, "c172code", by = "e16sex")

# add row and column percentages
data_tabulate(efc, "c172code", by = "e16sex", proportions = "row")
data_tabulate(efc, "c172code", by = "e16sex", proportions = "column")

# omit missing values
data_tabulate(
  efc$c172code,
  by = efc$e16sex,
  proportions = "column",
  remove_na = TRUE
)

# round percentages
out <- data_tabulate(efc, "c172code", by = "e16sex", proportions = "column")
print(out, digits = 0)

# coerce to data frames
result <- data_tabulate(efc, "c172code", by = "e16sex")
as.data.frame(result)
as.data.frame(result)$table
as.data.frame(result, add_total = TRUE)$table
\dontshow{\}) # examplesIf}
}
