#' Plot a list of model predictions
#' 
#' @param prediction Named list of matrices or data.frames, usually the output of a prediction function
#' as generated by \link{Xs}.
#' @param ... Further arguments going to \code{subset}. 
#' @details The data.frame being plotted has columns \code{time}, \code{value}, \code{name} and \code{condition}.
#'  
#' 
#' @return A plot object of class \code{ggplot}.
#' @export
#' @import ggplot2
plotPrediction <- function(prediction, ...) {

  prediction <- subset(wide2long.list(prediction), ...)
  
  ggplot2::ggplot(prediction, aes(x=time, y=value, group=condition, color=condition)) + facet_wrap(~name, scales="free") + geom_line()
  
}


#' Plot a list of model predictions and a list of data points in a combined plot
#' 
#' @param prediction Named list of matrices or data.frames, usually the output of a prediction function
#' as generated by \link{Xs}.
#' @param data Named list of data.frames as being used in \link{res}, i.e. with columns \code{name}, \code{time}, 
#' \code{value} and \code{sigma}.
#' @param ... Further arguments going to \code{subset}. 
#' @details The data.frame being plotted has columns \code{time}, \code{value}, \code{sigma},
#' \code{name} and \code{condition}.
#'  
#' 
#' @return A plot object of class \code{ggplot}.
#' @export
plotCombined <- function (prediction, data, ...) {
  
  mynames <- c("time", "name", "value", "sigma", "condition")
  prediction <- cbind(wide2long(prediction), sigma = NA)
  data <- lbind(data)
  
  
  prediction <- subset(prediction, ...)
  data <- subset(data, ...)
  
  ggplot2::ggplot(rbind(prediction[, mynames], data[, mynames]), 
                  aes(x = time, y = value, ymin = value - sigma, ymax = value + sigma, group = condition, color = condition)) + 
    facet_wrap(~name, scales = "free") + geom_line(data = prediction) + geom_point(data = data) + 
    geom_errorbar(data = data, width = 0)
  
}

#' Plot a list data points
#' 
#' @param data Named list of data.frames as being used in \link{res}, i.e. with columns \code{name}, \code{time}, 
#' \code{value} and \code{sigma}.
#' @param ... Further arguments going to \code{subset}. 
#' @details The data.frame being plotted has columns \code{time}, \code{value}, \code{sigma},
#' \code{name} and \code{condition}.
#'  
#' 
#' @return A plot object of class \code{ggplot}.
#' @export
plotData <- function (data, ...) {
  
  data <- subset(lbind(data), ...)
  ggplot2::ggplot(data, aes(x = time, y = value, ymin = value - sigma, 
                     ymax = value + sigma, group = condition, color = condition)) + 
    facet_wrap(~name, scales = "free") + geom_point() + geom_errorbar(width = 0)
}

#' Profile likelihood plot
#' 
#' @param ... Lists of profiles as being returned by \link{profile}.
#' @param maxvalue Numeric, the value where profiles are cut off.
#' @return A plot object of class \code{ggplot}.
#' @export
plotProfile <- function(..., maxvalue = 5) {
  
  
  arglist <- list(...)
  
  data <- do.call(rbind, lapply(1:length(arglist), function(i) {
    proflist <- arglist[[i]]
    subdata <- do.call(rbind, lapply(names(proflist), function(n) {
      
      values <- proflist[[n]][,1]
      zerovalue <- proflist[[n]]["out",1]
      parvalues <- proflist[[n]][,n]
      deltavalues <- values - zerovalue
      
      subset(data.frame(name = n, delta = deltavalues, par = parvalues, proflist = i), delta <= maxvalue)
      
    }))
    return(subdata)
  }))
  
  data$proflist <- as.factor(data$proflist)
  
  
  threshold <- c(1, 2.7, 3.84)
  
  ggplot2::ggplot(data, aes(x=par, y=delta, group=proflist, color=proflist)) + facet_wrap(~name, scales="free_x") + 
    geom_line() + geom_point(aes=aes(size=1), alpha=1/3) +
    geom_hline(yintercept=threshold, lty=2, color="gray") + 
    ylab(expression(paste("CL /", Delta*chi^2))) +
    scale_y_continuous(breaks=c(1, 2.7, 3.84), labels = c("68% / 1   ", "90% / 2.71", "95% / 3.84")) +
    xlab("parameter value")
  
  
}

#' Profile likelihood: plot of the parameter paths.
#' 
#' @param ... Lists of profiles as being returned by \link{profile}.
#' @param whichPar Character, the name of the parameter taken as a reference (x-axis)
#' @param sort Logical. If paths from different parameter profiles are plotted together, possible
#' combinations are either sorted or all combinations are taken as they are.
#' @return A plot object of class \code{ggplot}.
#' @export
plotPaths <- function(..., whichPar = NULL, sort = FALSE) {
  
  arglist <- list(...)
  
  data <- do.call(rbind, lapply(1:length(arglist), function(i) {
    # choose a proflist
    proflist <- arglist[[i]]
    if(is.null(whichPar)) whichPar <- names(proflist)
    subdata <- do.call(rbind, lapply(whichPar, function(n) {
      # chose a profile
      paths <- proflist[[n]][,-(1:4)]
      values <- proflist[[n]][,1]
      combinations <- expand.grid.alt(whichPar, colnames(paths))
      if(sort) combinations <- apply(combinations, 1, sort) else combinations <- apply(combinations, 1, identity)
      combinations <- combinations[,-which(combinations[1,] == combinations[2,])]
      combinations <- combinations[,!duplicated(paste(combinations[1,], combinations[2,]))]
      
      
        
      
      path.data <- do.call(rbind, lapply(1:dim(combinations)[2], function(j) {
        data.frame(chisquare = values, 
                   name = n,
                   proflist = i,
                   combination = paste(combinations[,j], collapse = " - "),
                   x = paths[,combinations[1,j]],
                   y = paths[,combinations[2,j]])
      }))
      
      return(path.data)
      
    }))
    
    return(subdata)
    
  }))
  
  data$proflist <- as.factor(data$proflist)
  
  
  ggplot2::ggplot(data, aes(x=x, y=y, group=interaction(name, proflist), color=name, lty=proflist)) + 
    facet_wrap(~combination, scales="free") + 
    geom_path() + geom_point(aes=aes(size=1), alpha=1/3) +
    xlab("parameter 1") + ylab("parameter 2") +
    scale_linetype_discrete(name = "profile\nlist") +
    scale_color_discrete(name = "profiled\nparameter")
  
}


