
#' Plots a three dimensional array
#' 
#' 1st margin sets line values, 2nd sets variables, 3rd quantiles.
#'
#' @param yarray 3 dimensional array to use for Y values
#' @param x numeric vector specifying x axis
#' @param grid Logical. Plot with a grid?
#' @param colvec color vector of same length as 2nd margin.
#' @param lwdvec lwd vector of same length as 2nd margin.
#' @param ltyvec lty vector of same length as 2nd margin.
#' @param typevec type vector of same length as 2nd margin.
#' @param plotcontrol list of arguments to pass to plot.
#' @param legend Logical. Draw a legend?
#' @param legendcontrol list of arguments to pass to \code{\link[graphics]{legend}}.
#' @param polygon Logical. Draw the uncertainty polygon?
#' @param polygonalpha Numeric, multiplier for alpha (transparency) of the 
#' uncertainty polygon.
#' @param polygoncontrol list of arguments to pass to \code{\link{ctPoly}}
#'
#' @return Nothing. Generates plots.
#' @export
#'
#' @examples
#' y <- ctStanTIpredeffects(ctstantestfit,plot=FALSE)
#' x<-ctstantestfit$data$tipreds[order(ctstantestfit$data$tipreds[,1]),1]
#' ctPlotArray(y,x)
ctPlotArray <- function(yarray,x,
  grid=TRUE,
  colvec='auto',lwdvec='auto',ltyvec='auto',typevec='auto',
  plotcontrol=list(ylab='Array values', xlab='X values',xaxs='i'),
  legend=TRUE,legendcontrol=list(x='topright'),
  polygon=TRUE, polygonalpha=.1,polygoncontrol=list(border=NA,steps=50)){
  
  separate=FALSE
  nvars<-dim(yarray)[2]
  
  plotcontrolpars <- c('ylab','xlab')
  plotcontroldefaults <- c('Array values','X values')
  for(i in 1:length(plotcontrolpars)) {
    if(is.null(plotcontrol[[plotcontrolpars[i]]])) plotcontrol[[plotcontrolpars[i]]] <- plotcontroldefaults[i]
  }
  
  legendcontrolpars <- c('x')
  legendcontroldefaults <- c('topright')
  for(i in 1:length(legendcontrolpars)) {
    if(is.null(legendcontrol[[legendcontrolpars[i]]])) legendcontrol[[legendcontrolpars[i]]] <- legendcontroldefaults[i]
  }
  
  polygoncontrolpars <- c('border')
  polygoncontroldefaults <- c(NA)
  for(i in 1:length(polygoncontrolpars)) {
    if(is.null(polygoncontrol[[polygoncontrolpars[i]]])) polygoncontrol[[polygoncontrolpars[i]]] <- polygoncontroldefaults[i]
  }
  
  if(all(ltyvec=='auto')) ltyvec = 1:nvars
  if(all(lwdvec=='auto')) lwdvec = rep(3,nvars)
  if(all(colvec=='auto')) colvec = rainbow(nvars)
  if(all(typevec=='auto')) typevec = rep('l',nvars)
  # if(all(mainvec=='auto')){
  #   if(separate) mainvec=dimnames(yarray)[[2]] else mainvec =rep(ifelse(is.null(plotcontrol$main),'',plotcontrol$main),nvars)
  # }
  
  plotargs<-plotcontrol
  plotargs$x <- x
  if(!separate && is.null(plotcontrol$ylim)) plotargs$ylim = range(yarray)
  plotargs$xlim = range(x,na.rm=TRUE)
  
  ctpolyargs<-polygoncontrol
  
  legargs<-legendcontrol
  
  
  #blank plot
  blankargs=plotargs
  blankargs$y=NA
  blankargs$x=NA
  do.call(plot,blankargs)
  if(grid) grid()
  
  #confidence
  if(polygon) {
    for(pari in c(1:dim(yarray)[2],dim(yarray)[2]:1)){
      ctpolyargs$col=adjustcolor(colvec[pari],alpha.f=max(c(.004,polygonalpha/ctpolyargs$steps)))
      ctpolyargs$x=plotargs$x
      ctpolyargs$y=yarray[,pari,2] #predict(loess(yarray[,pari,2]~ctpolyargs$x))
      ctpolyargs$yhigh = yarray[,pari,3] #predict(loess(yarray[,pari,3]~ctpolyargs$x))
      ctpolyargs$ylow = yarray[,pari,1]#predict(loess(yarray[,pari,1]~ctpolyargs$x))
      
      do.call(ctPoly,ctpolyargs) 
    }
  }
  
  for(pari in c(1:dim(yarray)[2],dim(yarray)[2]:1)){
    for(qi in 1:3){
      plotargs$y = yarray[,pari,qi]#predict(loess(yarray[,pari,qi]~plotargs$x))
      plotargs$col=colvec[pari]
      plotargs$lwd=ifelse(qi==2,lwdvec[pari],1)
      plotargs$lty=ltyvec[pari]
      if(qi!=2) plotargs$col =grDevices::adjustcolor(plotargs$col,alpha.f=.5)
      plotargs$type=typevec[pari]
      # plotargs$main=mainvec[pari]
      do.call(points,plotargs)
    }
    
    if(separate & legend) {
      legargs$legend=dimnames(yarray)[[2]][pari]
      legargs$col = plotargs$col
      legargs$lty = plotargs$lty
      legargs$text.col = plotargs$col
      legargs$lwd = plotargs$lwd
      do.call(graphics::legend,legargs) 
    }
  }
  
  if(!separate & legend) {
    legargs$legend=dimnames(yarray)[[2]]
    legargs$col = colvec
    legargs$lty = ltyvec
    legargs$text.col = colvec
    legargs$lwd = lwdvec
    do.call(graphics::legend,legargs) 
  }
  
  
}
