#' Compute all shortest distance between origin and destination nodes.
#' 
#' @param Graph  An object generated by cppRouting::makegraph() function.
#' @param from A vector of one or more vertices from which distances are calculated (origin).
#' @param to A vector of one or more vertices (destination).
#' @param allcores Logical. If TRUE, all cores are used.
#' @return Matrix of shortest distances.
#' @note get_distance_matrix() recursively perform Dijkstra algorithm for each 'from' nodes.
#' @examples 
#' #Data describing edges of the graph
#' edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4), 
#'                   to_vertex=c(1,3,2,4,4,5,1,3,5), 
#'                   cost=c(9,2,11,3,5,12,4,1,6))
#' #Get all nodes
#' nodes<-unique(c(edges$from_vertex,edges$to_vertex))
#'  
#' #Construct directed and undirected graph 
#' directed_graph<-makegraph(edges,directed=TRUE)
#' non_directed<-makegraph(edges,directed=FALSE)
#' 
#' #Get matrix of distance between all nodes in the two graphs
#' dir_dist<-get_distance_matrix(Graph=directed_graph, from=nodes, to=nodes, allcores=FALSE)
#' non_dir_dist<-get_distance_matrix(Graph=non_directed, from=nodes, to=nodes, allcores=FALSE)
#' print(dir_dist)
#' print(non_dir_dist)


get_distance_matrix<-function(Graph,from,to,allcores=FALSE){
  if (any(is.na(from))) stop("NAs are not allowed in origin/destination nodes")
  if (any(is.na(to))) stop("NAs are not allowed in origin/destination nodes")
  from<-as.character(from)
  to<-as.character(to)
  allnodes<-c(from,to)
  if (sum(allnodes %in% Graph$dict$ref)<length(allnodes)) stop("Some nodes are not in the graph")
  
  from_id<-Graph$dict$id[match(from,Graph$dict$ref)]
  to_id<-Graph$dict$id[match(to,Graph$dict$ref)]
  
  
  if (allcores==TRUE){
    numWorkers <- parallel::detectCores()
    cl <- parallel::makeCluster(numWorkers, type = "PSOCK")
    parallel::clusterEvalQ(cl = cl,library("cppRouting"))
    chunks <- parallel::splitIndices(length(from), ncl = numWorkers)
    mylist<-lapply(chunks,function(x) from_id[x])
    #mylist2<-lapply(chunks,function(x) to_id[x])
    res<-parallel::clusterMap(cl,Dijkstra_mat,dep=mylist,
                              MoreArgs = list(arr=to_id,gfrom=Graph$data$from,gto=Graph$data$to,gw=Graph$data$dist,NbNodes=Graph$nbnode))
    parallel::stopCluster(cl)
    res<-do.call(rbind,res)
    
  }
  else res<-Dijkstra_mat(Graph$data[,1],Graph$data[,2],Graph$data[,3],Graph$nbnode,from_id,to_id)
  
  rownames(res)<-from
  colnames(res)<-to
  return(res)
}
