% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxmeg.R
\name{coxmeg}
\alias{coxmeg}
\title{Fit a Cox mixed-effects model}
\usage{
coxmeg(outcome, corr, X = NULL, FID = NULL, eps = 1e-06,
  min_tau = 1e-04, max_tau = 5, order = 1, detap = TRUE,
  opt = "bobyqa", eigen = TRUE, dense = FALSE, solver = 1,
  spd = TRUE, verbose = TRUE, mc = 100, invchol = TRUE)
}
\arguments{
\item{outcome}{A matrix contains time (first column) and status (second column). The status is a binary variable (1 for events / 0 for censored).}

\item{corr}{A relatedness matrix. Can be a matrix or a 'dgCMatrix' class in the Matrix package. Must be symmetric positive definite or symmetric positive semidefinite.}

\item{X}{An optional matrix of the preidctors with fixed effects. Can be quantitative or binary values. Categorical variables need to be converted to dummy variables. Each row is a sample, and the predictors are columns.}

\item{FID}{An optional string vector of family ID. If provided, the data will be reordered according to the family ID.}

\item{eps}{An optional positive scalar indicating the tolerance in the optimization algorithm. Default is 1e-6.}

\item{min_tau}{An optional positive scalar indicating the lower bound in the optimization algorithm for the variance component \code{tau}. Default is 1e-4.}

\item{max_tau}{An optional positive scalar indicating the upper bound in the optimization algorithm for the variance component \code{tau} Default is 5.}

\item{order}{An optional integer scalar starting from 0. Only valid when \code{dense=FALSE}. It specifies the order of approximation used in the inexact newton method. Default is 1.}

\item{detap}{An optional logical scalar indicating whether to use approximation for log-determinant. Default is TRUE.}

\item{opt}{An optional logical scalar for the Optimization algorithm for tau. Can have the following values: 'bobyqa', 'Brent' or 'NM'. Default is 'bobyqa'.}

\item{eigen}{An optional logical scalar. Only effective when \code{dense=FALSE}. It indicates whether to use RcppEigen:LDLT to solve linear systems. Default is TRUE.}

\item{dense}{An optional logical scalar indicating whether the relatedness matrix is dense. Default is FALSE.}

\item{solver}{An optional bianry scalar taking either 1 or 2. Default is 1. See details.}

\item{spd}{An optional logical value indicating whether the relatedness matrix is symmetric positive definite. Default is TRUE. See details.}

\item{verbose}{An optional logical scalar indicating whether to print additional messages. Default is TRUE.}

\item{mc}{An optional integer scalar specifying the number of Monte Carlo samples used for approximating the log-determinant. Only valid when dense=TRUE and detap=TRUE. Default is 100.}

\item{invchol}{An optional logical value. Only effective when \code{dense=FALSE}. If TRUE, sparse Cholesky decomposition is used to compute the inverse of the relatedness matrix. Otherwise, sparse LU is used.}
}
\value{
beta: The estimated coefficient for each predictor in X.

HR: The estimated HR for each predictor in X.

sd_beta: The estimated standard error of beta.

iter: The number of iterations until convergence.

tau: The estimated variance component.

int_ll: The marginal likelihood (-2*log(lik)) of tau evaluated at the estimate of tau.

rank: The rank of the relatedness matrix.

nsam: Actual sample size.
}
\description{
\code{coxmeg} returns estimates of the variance component, the HRs and p-values for the predictors.
}
\section{About \code{spd}}{

When \code{spd=TRUE}, the relatedness matrix is treated as SPD. If the matrix is SPSD or not sure, use \code{spd=FALSE}.
}

\section{About \code{solver}}{

When solver=1/solver=2, Cholesky decompositon/PCG is used to solve the linear system. However, when \code{dense=FALSE} and \code{eigen=FALSE}, the solve function in the Matrix package is used regardless of \code{solver}. When \code{dense=TRUE}, it is recommended to set \code{solver=2} to have better computational performance.
}

\section{About \code{invchol}}{

Cholesky decomposition using \code{invchol=TRUE} is generally (but not always) much faster to invert a relatedness matrix (e.g., a block-diagonal matrix). But for some types of sparse matrices (e.g., a banded AR(1) matrix with rho=0.9), it sometimes can be very slow. In such cases, \code{invchol=FALSE} is can be used.
}

\examples{
library(Matrix)
library(MASS)
library(coxmeg)

## simulate a block-diagonal relatedness matrix
tau_var <- 0.2
n_f <- 100
mat_list <- list()
size <- rep(10,n_f)
offd <- 0.5
for(i in 1:n_f)
{
  mat_list[[i]] <- matrix(offd,size[i],size[i])
  diag(mat_list[[i]]) <- 1
}
sigma <- as.matrix(bdiag(mat_list))
n <- nrow(sigma)

## simulate random effects and outcomes
x <- mvrnorm(1, rep(0,n), tau_var*sigma)
myrates <- exp(x-1)
y <- rexp(n, rate = myrates)
cen <- rexp(n, rate = 0.02 )
ycen <- pmin(y, cen)
outcome <- cbind(ycen,as.numeric(y <= cen))

## fit a Cox mixed-effects model
re = coxmeg(outcome,sigma,order=1,eigen=TRUE,dense=FALSE)
re
}
\keyword{Cox}
\keyword{mixed-effects}
\keyword{model}
