#' Uniform parameter
#'
#' Parameter with uniform distribution for hyperparameter optimization
#' @export
#' @param name Name of the parameter, must match the input to `eval_func`.
#' @param lower Lower bound of the parameter
#' @param upper Upper bound of the parameter
#' @return Returns an R6 class generated by R6_par_unif.
#' @examples
#' p1 <- par_unif('x1', 1, 10)
#' class(p1)
#' print(p1)
par_unif <- function(name, lower, upper) {
  R6_par_unif$new(
    name=name,
    lower=lower,
    upper=upper
  )
}

#' R6 class for Uniform parameter
#'
#' Parameter with uniform distribution for hyperparameter optimization
#' @export
#' @field name Name of the parameter, must match the input to `eval_func`.
#' @field lower Lower bound of the parameter
#' @field upper Upper bound of the parameter
# @field fromraw Function to convert from raw scale to transformed scale
# @field toraw Function to convert from transformed scale to raw scale
#' @field ggtrans Transformation for ggplot, see ggplot2::scale_x_continuous()
R6_par_unif <- R6::R6Class(
  # par_unif ----
  classname="par_unif",
  inherit = R6_par_hype,
  public=list(
    name=NULL,
    lower=NULL,
    upper=NULL,
    #' @description Function to convert from raw scale to transformed scale
    #' @param x Value of raw scale
    fromraw=function(x) {x}, #identity,
    #' @description Function to convert from transformed scale to raw scale
    #' @param x Value of transformed scale
    toraw= function(x) {x}, #identity,
    #' @description Generate values in the raw space based on quantiles.
    #' @param q In [0,1].
    # generate = function(q) {
    #   self$toraw(self$fromraw(self$lower) + q * (self$fromraw(self$upper) - self$fromraw(self$lower)))
    # },
    generate = function(q) {
      stopifnot(is.numeric(q), q>=0, q<=1)
      self$lower + q * (self$upper - self$lower)
    },
    #' @description Check if input is valid for parameter
    #' @param x Parameter value
    isvalid = function(x) {
      is.numeric(x) &
        (x >= self$lower) &
        (x <= self$upper)
    },
    #' @description Convert this to a parameter for the
    #' mixopt R package.
    #' @param raw_scale Should it be on the raw scale?
    convert_to_mopar = function(raw_scale=FALSE) {
      mixopt::mopar_cts(lower=self$lower,
                        upper=self$upper)
    },
    ggtrans="identity", # ggplot trans to give to scale_x_continuous
    # fromraw=NULL,
    # toraw= NULL,
    # ggtrans=NULL, # ggplot trans to give to scale_x_continuous
    #' @description Create a hyperparameter with uniform distribution
    #' @param name Name of the parameter, must match the input to `eval_func`.
    #' @param lower Lower bound of the parameter
    #' @param upper Upper bound of the parameter
    initialize = function(name, lower, upper) {
      self$name <- name
      self$lower <- lower
      self$upper <- upper
      # These can't be defined above, gave error, has to be in init
      # self$fromraw <- identity
      # self$toraw <- identity
      # self$ggtrans <- "identity" # ggplot trans to give to scale_x_continuous
    },
    #' @description Print details of the object.
    #' @param ... not used,
    print = function(...) {
      s <- paste0("hype par_unif(name = ", self$name, ", lower = ", self$lower,
                  ", upper = ", self$upper, ")")
      cat(s)
      invisible(self)
    }
  )
)
if (F) {
  p1 <- par_unif('x1', 0, 2)
  class(p1)
}

if (F) {
  # @rdname par_unif
  # @export
  par_unif2 <- function(name, lower, upper) {
    par_unif(name=name,
                 lower=lower,
                 upper=upper)
  }
}
