\name{ccaGrid}
\alias{ccaGrid}
\alias{CCAgrid}
\alias{print.cca}
\title{(Robust) CCA via alternating series of grid searches}
\usage{
  ccaGrid(x, y, k = 1,
    method = c("spearman", "kendall", "quadrant", "M", "pearson"),
    control = list(...), nIterations = 10, nAlternate = 10,
    nGrid = 25, select = NULL, tol = 1e-06,
    standardize = TRUE, fallback = FALSE, seed = NULL, ...)

  CCAgrid(x, y, k = 1,
    method = c("spearman", "kendall", "quadrant", "M", "pearson"),
    maxiter = 10, maxalter = 10, splitcircle = 25,
    select = NULL, zero.tol = 1e-06, standardize = TRUE,
    fallback = FALSE, seed = NULL, ...)
}
\arguments{
  \item{x,y}{each can be a numeric vector, matrix or data
  frame.}

  \item{k}{an integer giving the number of canonical
  variables to compute.}

  \item{method}{a character string specifying the
  correlation functional to maximize.  Possible values are
  \code{"spearman"} for the Spearman correlation,
  \code{"kendall"} for the Kendall correlation,
  \code{"quadrant"} for the quadrant correlation,
  \code{"M"} for the correlation based on a bivariate
  M-estimator of location and scatter with a Huber loss
  function, or \code{"pearson"} for the classical Pearson
  correlation (see \code{\link{corFunctions}}).}

  \item{control}{a list of additional arguments to be
  passed to the specified correlation functional.  If
  supplied, this takes precedence over additional arguments
  supplied via the \code{\dots} argument.}

  \item{nIterations,maxiter}{an integer giving the maximum
  number of iterations.}

  \item{nAlternate,maxalter}{an integer giving the maximum
  number of alternate series of grid searches in each
  iteration.}

  \item{nGrid,splitcircle}{an integer giving the number of
  equally spaced grid points on the unit circle to use in
  each grid search.}

  \item{select}{optional; either an integer vector of
  length two or a list containing two index vectors.  In
  the first case, the first integer gives the number of
  variables of \code{x} to be randomly selected for
  determining the order of the variables of \code{y} in the
  corresponding series of grid searches, and vice versa for
  the second integer.  In the latter case, the first list
  element gives the indices of the variables of \code{x} to
  be used for determining the order of the variables of
  \code{y}, and vice versa for the second integer (see
  \dQuote{Details}).}

  \item{tol,zero.tol}{a small positive numeric value to be
  used for determining convergence.}

  \item{standardize}{a logical indicating whether the data
  should be (robustly) standardized.}

  \item{fallback}{logical indicating whether a fallback
  mode for robust standardization should be used.  If a
  correlation functional other than the Pearson correlation
  is maximized, the first attempt for standardizing the
  data is via median and MAD.  In the fallback mode,
  variables whose MADs are zero (e.g., dummy variables) are
  standardized via mean and standard deviation.  Note that
  if the Pearson correlation is maximized, standardization
  is always done via mean and standard deviation.}

  \item{seed}{optional initial seed for the random number
  generator (see \code{\link{.Random.seed}}).  This is only
  used if \code{select} specifies the numbers of variables
  of each data set to be randomly selected for determining
  the order of the variables of the respective other data
  set.}

  \item{\dots}{additional arguments to be passed to the
  specified correlation functional.  Currently, this is
  only relevant for the M-estimator.  For Spearman, Kendall
  and quadrant correlation, consistency at the normal model
  is always forced.}
}
\value{
  An object of class \code{"cca"} with the following
  components:

  \item{cor}{a numeric vector giving the canonical
  correlation measures.}

  \item{A}{a numeric matrix in which the columns contain
  the canonical vectors for \code{x}.}

  \item{B}{a numeric matrix in which the columns contain
  the canonical vectors for \code{y}.}

  \item{centerX}{a numeric vector giving the center
  estimates used in standardization of \code{x}.}

  \item{centerY}{a numeric vector giving the center
  estimates used in standardization of \code{y}.}

  \item{scaleX}{a numeric vector giving the scale estimates
  used in standardization of \code{x}.}

  \item{scaleY}{a numeric vector giving the scale estimates
  used in standardization of \code{y}.}

  \item{call}{the matched function call.}
}
\description{
  Perform canoncial correlation analysis via projection
  pursuit based on alternating series of grid searches in
  two-dimensional subspaces of each data set, with a focus
  on robust and nonparametric methods.
}
\details{
  The algorithm is based on alternating series of grid
  searches in two-dimensional subspaces of each data set.
  In each grid search, \code{nGrid} grid points on the unit
  circle in the corresponding plane are obtained, and the
  directions from the center to each of the grid points are
  examined.  In the first iteration, equispaced grid points
  in the interval \eqn{[-\pi/2, \pi/2)}{[-pi/2, pi/2)} are
  used.  In each subsequent iteration, the angles are
  halved such that the interval \eqn{[-\pi/4,
  \pi/4)}{[-pi/4, pi/4)} is used in the second iteration
  and so on.  If only one data set is multivariate, the
  algorithm simplifies to iterative grid searches in
  two-dimensional subspaces of the corresponding data set.

  In the basic algorithm, the order of the variables in a
  series of grid searches for each of the data sets is
  determined by the average absolute correlations with the
  variables of the respective other data set.  Since this
  requires to compute the full \eqn{(p \times q)}{(p x q)}
  matrix of absolute correlations, where \eqn{p} denotes
  the number of variables of \code{x} and \eqn{q} the
  number of variables of \code{y}, a faster modification is
  available as well.  In this modification, the average
  absolute correlations are computed over only a subset of
  the variables of the respective other data set.  It is
  thereby possible to use randomly selected subsets of
  variables, or to specify the subsets of variables
  directly.

  Note that also the data sets are ordered according to the
  maximum average absolute correlation with the respective
  other data set to ensure symmetry of the algorithm.

  For higher order canonical correlations, the data are
  first transformed into suitable subspaces.  Then the
  alternate grid algorithm is applied to the reduced data
  and the results are back-transformed to the original
  space.
}
\note{
  \code{CCAgrid} is a simple wrapper function for
  \code{ccaGrid} for more compatibility with package
  \pkg{pcaPP} concerning function and argument names.
}
\examples{
## generate data
library("mvtnorm")
set.seed(1234)  # for reproducibility
p <- 3
q <- 2
m <- p + q
sigma <- 0.5^t(sapply(1:m, function(i, j) abs(i-j), 1:m))
xy <- rmvnorm(100, sigma=sigma)
x <- xy[, 1:p]
y <- xy[, (p+1):m]

## Spearman correlation
ccaGrid(x, y, method = "spearman")

## Pearson correlation
ccaGrid(x, y, method = "pearson")
}
\author{
  Andreas Alfons
}
\seealso{
  \code{\link{ccaProj}}, \code{\link{maxCorGrid}},
  \code{\link{corFunctions}}
}
\keyword{multivariate}
\keyword{robust}

